/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_line.h"

extern void
item_line_assign(
  struct item_line*const                o_line,
  struct item_method*const              o_method)
{

  memset(o_line, 0, sizeof(*o_line));

  (*o_line).m_pos_x1= 10.0;
  (*o_line).m_pos_y1= 10.0;
  (*o_line).m_pos_x2= 100.0;
  (*o_line).m_pos_y2= 10.0;
  (*o_line).m_thickness= 1;
  (*o_line).m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, line);

  return;
}

extern void
item_line_copy(
  struct item_line *const               o_line,
  struct item_line const*const          i_line)
{

  memcpy(o_line, i_line, sizeof(*i_line));

  return;
}

extern void
item_line_discharge(
  struct item_line*const                io_line)
{
  memset(io_line, 0, sizeof(*io_line));
}

extern void
item_line_draw(
  struct item_line*const                i_line,
  struct item_draw_context const*const  i_ctx)
{

  cairo_new_path((*i_ctx).m_cr);
  cairo_save((*i_ctx).m_cr);
  cairo_scale((*i_ctx).m_cr, (*i_ctx).m_scale, (*i_ctx).m_scale);

  cairo_set_source_rgba(
    (*i_ctx).m_cr,
    (*i_line).m_color.red,
    (*i_line).m_color.green,
    (*i_line).m_color.blue,
    (*i_line).m_color.alpha);

  item_common_cairo_dash_set((*i_ctx).m_cr, (*i_line).m_style);
  cairo_set_line_width((*i_ctx).m_cr, (*i_line).m_thickness);

  cairo_move_to((*i_ctx).m_cr, (*i_line).m_pos_x1, (*i_line).m_pos_y1);
  cairo_line_to((*i_ctx).m_cr, (*i_line).m_pos_x2, (*i_line).m_pos_y2);
  cairo_stroke((*i_ctx).m_cr);

  if ((*i_line).m_double_line)
  {
    cairo_move_to((*i_ctx).m_cr, (*i_line).m_pos_x1, 3.00+(*i_line).m_pos_y1);
    cairo_line_to((*i_ctx).m_cr, (*i_line).m_pos_x2, 3.00+(*i_line).m_pos_y2);
    cairo_stroke((*i_ctx).m_cr);
  }

  cairo_restore((*i_ctx).m_cr);

  return;
}

extern void
item_line_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  double const                          i_scale,
  void const*const                      i_object)
{
  struct item_line const*               l_line;

  l_line= (struct item_line const*)i_object;
  (*o_bbox).m_pos_x= i_scale * (*l_line).m_pos_x1;
  (*o_bbox).m_pos_y= i_scale * (*l_line).m_pos_y1;
  (*o_bbox).m_len_x= i_scale * ((*l_line).m_pos_x2 - (*l_line).m_pos_x1);
  (*o_bbox).m_len_y= i_scale * ((*l_line).m_pos_y2 - (*l_line).m_pos_y1);

  return;
}

extern void
item_line_get_position(
  double*const                          o_pos_x,
  double*const                          o_pos_y,
  struct item_line const*const          i_line)
{

  (*o_pos_x)= (*i_line).m_pos_x1;
  (*o_pos_y)= (*i_line).m_pos_y1;

  return;
}

extern void
item_line_get_rotation(
  double*const                          o_rotation,
  struct item_line const*const          i_line)
{

  (*o_rotation)= 0.0;

  return;
}

extern void
item_line_get_size(
  double*const                          o_len_x,
  double*const                          o_len_y,
  struct item_line const*const          i_line)
{

  (*o_len_x)= (*i_line).m_pos_x2 - (*i_line).m_pos_x1;
  (*o_len_y)= (*i_line).m_pos_y2 - (*i_line).m_pos_y1;

  return;
}

extern void
item_line_get_shear(
  double*const                          o_shear_x,
  double*const                          o_shear_y,
  struct item_line const*const          i_line)
{
  
  (*o_shear_x)= 0.0;
  (*o_shear_y)= 0.0;

  return;
}

extern int
item_line_in_event(
  struct item_line const*const          i_line,
  cairo_t*                              io_cr,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= item_common_line_in_event(i_ctx, io_cr, (struct geom_line*)i_line);

  return l_hit;
}

extern void
item_line_move(
  struct item_line*const                io_line,
  struct geom_point const*const         i_event)
{

  (*io_line).m_pos_x1+= (*i_event).m_pos_x;
  (*io_line).m_pos_y1+= (*i_event).m_pos_y;
  (*io_line).m_pos_x2+= (*i_event).m_pos_x;
  (*io_line).m_pos_y2+= (*i_event).m_pos_y;

  return;
}

extern void
item_line_normalize(
  struct item_line*const                io_line)
{
  gdouble                               l_swap;

  do
  {

    if ((*io_line).m_pos_x1 > (*io_line).m_pos_x2)
    {
      l_swap= (*io_line).m_pos_x1;
      (*io_line).m_pos_x1= (*io_line).m_pos_x2;
      (*io_line).m_pos_x2= l_swap;

      l_swap= (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= (*io_line).m_pos_y2;
      (*io_line).m_pos_y2= l_swap;
    }

  }while(0);

  return;
}

extern void
item_line_set_position(
  struct item_line*const                io_line,
  double const                          i_pos_x,
  double const                          i_pos_y)
{

  (*io_line).m_pos_x1= i_pos_x;
  (*io_line).m_pos_y1= i_pos_y;

  return;
}

extern void
item_line_set_rotation(
  struct item_line*const                io_line,
  double                                i_rotation)
{
  return;
}

extern void
item_line_set_size(
  struct item_line*const                io_line,
  double                                i_len_x,
  double                                i_len_y)
{

  (*io_line).m_pos_x2= (*io_line).m_pos_x1 + i_len_x;
  (*io_line).m_pos_y2= (*io_line).m_pos_y1 + i_len_y;

  return;
}

extern void
item_line_set_shear(
  struct item_line*const                io_line,
  double const                          i_shear_x,
  double const                          i_shear_y)
{
  return;
}

extern int
item_line_read(
  GError**                              o_error,
  struct item_line*const                io_line,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x1:
          (*io_line).m_pos_x1= xml_convert_double(l_value);
          break;
        case element_tag_pos_y1:
          (*io_line).m_pos_y1= xml_convert_double(l_value);
          break;
        case element_tag_pos_x2:
          (*io_line).m_pos_x2= xml_convert_double(l_value);
          break;
        case element_tag_pos_y2:
          (*io_line).m_pos_y2= xml_convert_double(l_value);
          break;
        case element_tag_style:
          (*io_line).m_style= item_common_get_line_style(l_value);
          break;
        case element_tag_thickness:
          (*io_line).m_thickness= g_ascii_strtod(l_value, 0);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_line).m_color= l_rgba;
          }
          break;
        case element_tag_doubleline:
          l_value= xml_lookup_attribute_value(l_xml, "on");
          (*io_line).m_double_line= xml_convert_boolean(l_value);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_line_resize(
  struct item_line*const                io_line,
  struct item_resize_event const*const  i_ctx)
{
  enum event_type                       l_type;

  l_type= (*i_ctx).m_type;

  if ((*io_line).m_pos_y1 > (*io_line).m_pos_y2)
  {
    switch(l_type)
    {
      case event_resize_north:
        l_type= event_resize_south;
        break;
      case event_resize_south:
        l_type= event_resize_north;
        break;
      case event_resize_north_west:
        l_type= event_resize_south_west;
        break;
      case event_resize_north_east:
        l_type= event_resize_south_east;
        break;
      case event_resize_south_east:
        l_type= event_resize_north_east;
        break;
      case event_resize_south_west:
        l_type= event_resize_north_west;
        break; 
      default:
        break;
    }
  }
  
  switch(l_type)
  {
    case event_resize_east:
      (*io_line).m_pos_x2+= (*i_ctx).m_delta_x;
      break;
    case event_resize_north:
      (*io_line).m_pos_y1+= (*i_ctx).m_delta_y;
      break;
    case event_resize_north_east:
      (*io_line).m_pos_y1+= (*i_ctx).m_delta_y;
      (*io_line).m_pos_x2+= (*i_ctx).m_delta_x;
      break;
    case event_resize_north_west:
      (*io_line).m_pos_y1+= (*i_ctx).m_delta_y;
      (*io_line).m_pos_x1+= (*i_ctx).m_delta_x;
      break;
    case event_resize_south:
      (*io_line).m_pos_y2+= (*i_ctx).m_delta_y;
      break;
    case event_resize_south_east:
      (*io_line).m_pos_y2+= (*i_ctx).m_delta_y;
      (*io_line).m_pos_x2+= (*i_ctx).m_delta_x;
      break;
    case event_resize_south_west:
      (*io_line).m_pos_y2+= (*i_ctx).m_delta_y;
      (*io_line).m_pos_x1+= (*i_ctx).m_delta_x;
      break;
    case event_resize_west:
      (*io_line).m_pos_x1+= (*i_ctx).m_delta_x;
      break;
    default:
      break;
  }

  return;
}

extern void
item_line_rotate(
  struct item_line*const                io_line,
  double const                          i_rotation)
{
}
  
extern void
item_line_shear(
  struct item_line*const                io_line,
  double const                          i_delta_x,
  double const                          i_delta_y)
{
}

extern int
item_line_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_line const*const          i_line)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"line\">\n");
  fprintf(io_fp, "\t\t\t\t<pos_x1>%s</pos_x1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_x1));
  fprintf(io_fp, "\t\t\t\t<pos_y1>%s</pos_y1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_y1));
  fprintf(io_fp, "\t\t\t\t<pos_x2>%s</pos_x2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_x2));
  fprintf(io_fp, "\t\t\t\t<pos_y2>%s</pos_y2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_y2));
  l_text= gdk_rgba_to_string(&(*i_line).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t<style>%s</style>\n", item_common_get_line_style_text((*i_line).m_style));
  fprintf(io_fp, "\t\t\t\t<thickness>%s</thickness>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_thickness));
  fprintf(io_fp, "\t\t\t\t<double_line on=\"%s\"/>\n", ((*i_line).m_double_line ? "true" : "false"));
  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
