/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "pixbuf.h"
#include "item_common_base.h"

extern void
pixbuf_get_attributes(
  struct pixbuf_attribute*const         o_attr,
  GdkPixbuf const*const                 i_pixbuf)
{

  memset(o_attr, 0, sizeof(*o_attr));
  (*o_attr).m_colorspace= gdk_pixbuf_get_colorspace(i_pixbuf);
  (*o_attr).m_channels= gdk_pixbuf_get_n_channels(i_pixbuf);
  (*o_attr).m_has_alpha= gdk_pixbuf_get_has_alpha(i_pixbuf);
  (*o_attr).m_bits_per_sample= gdk_pixbuf_get_bits_per_sample(i_pixbuf);
  (*o_attr).m_width= gdk_pixbuf_get_width(i_pixbuf);
  (*o_attr).m_height= gdk_pixbuf_get_height(i_pixbuf);
  (*o_attr).m_rowstride= gdk_pixbuf_get_rowstride(i_pixbuf);
  (*o_attr).m_byte_length= gdk_pixbuf_get_byte_length(i_pixbuf);

  return;
}

static GdkColorspace 
pixbuf_get_colorspace(
  char const*                           i_colorspace)
{
  return GDK_COLORSPACE_RGB;
}

static char const*
pixbuf_get_colorspace_text(
  GdkColorspace const                   i_colorspace)
{
  return "RGB";
}

extern int
pixbuf_new_from_blob(
  GError**                              o_error,
  GdkPixbuf **                          o_pixbuf,
  void const*                           i_blob,
  size_t const                          i_size)
{
  GError*                               l_error;
  int                                   l_exit;
  GdkPixbufLoader*                      l_loader;
  GdkPixbuf*                            l_pixbuf;

  (*o_pixbuf)= 0;
  l_error= 0;
  l_exit= 0;
  l_loader= 0;
  l_pixbuf= 0;

  do
  {

    if (0 == i_blob || 0 == i_size)
    {
      break;
    }

    l_loader= gdk_pixbuf_loader_new();

    gdk_pixbuf_loader_write(l_loader, i_blob, i_size, &l_error);
    gdk_pixbuf_loader_close(l_loader, 0);

    if (l_error)
    {
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_pixbuf= gdk_pixbuf_loader_get_pixbuf(l_loader);

    if (0 == l_pixbuf)
    {
      break;
    }

    (*o_pixbuf)= l_pixbuf;


  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  if (l_loader)
  {
    g_object_unref(l_loader);
  }

  return l_exit;
}

static int
pixbuf_read_data(
  GError**                              o_error,
  guchar**                              o_encode,   
  struct pixbuf_attribute*const         io_attr,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  (*o_encode)= 0;
  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_colorspace:
          (*io_attr).m_colorspace= pixbuf_get_colorspace(l_value);
          break;
        case element_tag_channels:
          (*io_attr).m_channels= xml_convert_ulong(l_value);
          break;
        case element_tag_has_alpha:
          (*io_attr).m_has_alpha= xml_convert_ulong(l_value);
          break;
        case element_tag_bits_per_sample:
          (*io_attr).m_bits_per_sample= xml_convert_ulong(l_value);
          break;
        case element_tag_width:
          (*io_attr).m_width= xml_convert_ulong(l_value);
          break;
        case element_tag_height:
          (*io_attr).m_height= xml_convert_ulong(l_value);
          break;
        case element_tag_rowstride:
          (*io_attr).m_rowstride= xml_convert_ulong(l_value);
          break;
        case element_tag_length:
          (*io_attr).m_byte_length= xml_convert_ulong(l_value);
          break;
        case element_tag_encode:
          (*o_encode)= (guchar*)g_strdup(l_value);
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static void
pixbuf_destroy_notify(
  guchar*                               i_pixels,
  gpointer                              i_data)
{
  g_free(i_pixels);
  return;
}

extern int
pixbuf_read(
  GError**                              o_error,
  GdkPixbuf **                          o_pixbuf,
  struct xml *const                     i_xml)
{
  int                                   l_exit;
  GError*                               l_error;
  struct pixbuf_attribute               l_attr;
  guchar*                               l_encode;
  guchar*                               l_compr;
  gsize                                 l_compr_len;
  guchar*                               l_pixels;
  gsize                                 l_pixels_len;
  int                                   l_rc;

  (*o_pixbuf)= 0;
  memset(&l_attr, 0, sizeof(l_attr));
  l_compr= 0;
  l_pixels= 0;
  l_exit= 0;
  l_error= 0;

  do
  {

    l_exit= pixbuf_read_data(&l_error, &l_encode, &l_attr, i_xml);

    if (l_exit)
    {
      break;
    }

    if (0 == l_encode)
    {
      break;
    }

    l_compr= g_base64_decode((gchar*)l_encode, &l_compr_len);

    l_pixels_len= l_attr.m_byte_length;
    l_pixels= (guchar*)g_malloc0(l_pixels_len);
    l_rc= uncompress(l_pixels, &l_pixels_len, l_compr, l_compr_len);

    if (Z_OK != l_rc)
    {
      break;
    }

    (*o_pixbuf)= gdk_pixbuf_new_from_data(
      l_pixels,
      l_attr.m_colorspace,
      l_attr.m_has_alpha,
      l_attr.m_bits_per_sample,
      l_attr.m_width,
      l_attr.m_height,
      l_attr.m_rowstride,
      pixbuf_destroy_notify,
      0);

  }while(0);

  g_free(l_compr);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
pixbuf_write(
  FILE*                                 io_fp,
  GdkPixbuf const*const                 i_pixbuf)
{
  struct pixbuf_attribute               l_attr;
  int                                   l_exit;
  guchar*                               l_pixels;
  gchar*                                l_encode;
  guchar*                               l_compr;
  unsigned long                         l_compr_size;

  l_exit= 0;

  pixbuf_get_attributes(&l_attr, i_pixbuf);

  fprintf(io_fp, "\t\t\t\t<data>\n");
  fprintf(io_fp, "\t\t\t\t\t<colorspace>%s</colorspace>\n", pixbuf_get_colorspace_text(l_attr.m_colorspace));
  fprintf(io_fp, "\t\t\t\t\t<channels>%ld</channels>\n", l_attr.m_channels);
  fprintf(io_fp, "\t\t\t\t\t<has_alpha>%ld</has_alpha>\n", l_attr.m_has_alpha);
  fprintf(io_fp, "\t\t\t\t\t<bits_per_sample>%ld</bits_per_sample>\n", l_attr.m_bits_per_sample);
  fprintf(io_fp, "\t\t\t\t\t<width>%ld</width>\n", l_attr.m_width);
  fprintf(io_fp, "\t\t\t\t\t<height>%ld</height>\n", l_attr.m_height);
  fprintf(io_fp, "\t\t\t\t\t<rowstride>%ld</rowstride>\n", l_attr.m_rowstride);
  fprintf(io_fp, "\t\t\t\t\t<length>%ld</length>\n", l_attr.m_byte_length);

  l_pixels= gdk_pixbuf_get_pixels(i_pixbuf);

  l_compr_size= compressBound(l_attr.m_byte_length);
  l_compr= g_malloc0(l_compr_size);

  compress(l_compr, &l_compr_size, l_pixels, l_attr.m_byte_length);

  l_encode= g_base64_encode(l_compr, l_compr_size);

  fprintf(io_fp, "\t\t\t\t\t<encode>%s</encode>\n", l_encode);

  g_free(l_encode);
  g_free(l_compr);

  fprintf(io_fp, "\t\t\t\t</data>\n");

  return l_exit;
}
