/**
 * ProgressDialog.java
 *
 * Copyright (C) 2010 imedias
 *
 * This file is part of JBackpack.
 *
 * JBackpack is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * JBackpack is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * Created on 14.06.2010, 09:55:31
 */
package ch.fhnw.jbackpack;

import ch.fhnw.util.ProcessExecutor;
import java.awt.Window;
import javax.swing.Icon;

/**
 * a dialog for indeterminate progress messages
 *
 * @author Ronny Standtke <Ronny.Standtke@gmx.net>
 */
public class ProgressDialog extends javax.swing.JDialog {

    private final Window parent;
    private final ProcessExecutor processExecutor;

    /**
     * Creates new form ProgressDialog
     *
     * @param parent the parent frame
     * @param processExecutor the ProcessExecutor to be used when pressing the
     * cancel button
     */
    public ProgressDialog(Window parent, ProcessExecutor processExecutor) {
        super(parent, ModalityType.APPLICATION_MODAL);
        this.parent = parent;
        this.processExecutor = processExecutor;
        initComponents();
        label.setIcon(IconManager.INFORMATION_ICON);
    }

    /**
     * sets the icon for the label
     *
     * @param icon the icon for the label
     */
    public void setIcon(Icon icon) {
        label.setIcon(icon);
    }

    /**
     * sets the icon for the special icon label
     *
     * @param icon the icon for the special icon label
     */
    public void setSpecialIcon(Icon icon) {
        specialIconLabel.setIcon(icon);
    }

    /**
     * makes the cancel button visible or invisible
     *
     * @param visible if <tt>true</tt>, the cancel button is made visible; if
     * <tt>false</tt> the cancel button is made invisible
     */
    public void setCancelButtonVisible(boolean visible) {
        cancelButton.setVisible(visible);
        pack();
        setLocationRelativeTo(parent);
    }

    /**
     * sets the message to display, packs and centers the dialog
     *
     * @param message the message to display
     */
    public void setMessage(String message) {
        label.setText(message);
        pack();
        setLocationRelativeTo(parent);
    }

    /**
     * Sets the indeterminate property of the progress bar
     *
     * @param indeterminate true if the progress bar should change to
     * indeterminate mode; false if it should revert to normal.
     */
    public void setIndeterminate(boolean indeterminate) {
        progressBar.setIndeterminate(indeterminate);
        progressBar.setStringPainted(!indeterminate);
        pack();
        setLocationRelativeTo(parent);
    }

    /**
     * Sets the progress bar's current value and string.
     *
     * @param progress the progress value
     */
    public void setProgress(int progress) {
        progressBar.setValue(progress);
        progressBar.setString(progress + "%");
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        label = new javax.swing.JLabel();
        specialIconLabel = new javax.swing.JLabel();
        progressBar = new javax.swing.JProgressBar();
        cancelButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("ch/fhnw/jbackpack/Strings"); // NOI18N
        setTitle(bundle.getString("ProgressDialog.title")); // NOI18N
        getContentPane().setLayout(new java.awt.GridBagLayout());

        label.setText(bundle.getString("Removing_Selected_Backups")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 0, 10);
        getContentPane().add(label, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.insets = new java.awt.Insets(10, 0, 0, 0);
        getContentPane().add(specialIconLabel, gridBagConstraints);

        progressBar.setIndeterminate(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 10);
        getContentPane().add(progressBar, gridBagConstraints);

        cancelButton.setText(bundle.getString("ProgressDialog.cancelButton.text")); // NOI18N
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 0);
        getContentPane().add(cancelButton, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        processExecutor.destroy();
        // disable cancelButton so that no other processes are cancelled by
        // accident
        cancelButton.setEnabled(false);
    }//GEN-LAST:event_cancelButtonActionPerformed
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cancelButton;
    private javax.swing.JLabel label;
    private javax.swing.JProgressBar progressBar;
    private javax.swing.JLabel specialIconLabel;
    // End of variables declaration//GEN-END:variables
}
