/**
 * DatabaseSyncDialog.java
 *
 * Copyright (C) 2010 imedias
 *
 * This file is part of JBackpack.
 *
 * JBackpack is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * JBackpack is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * Created on 05.09.2010, 17:36:36
 */
package ch.fhnw.jbackpack.chooser;

import java.awt.CardLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.ResourceBundle;
import java.util.TimeZone;
import javax.swing.JProgressBar;
import javax.swing.Timer;

/**
 * A dialog shown when syncing the file database
 *
 * @author Ronny Standtke <Ronny.Standtke@gmx.net>
 */
public class DatabaseSyncDialog extends javax.swing.JDialog {

    private final static DateFormat DATE_FORMAT =
            DateFormat.getDateTimeInstance();
    private final static NumberFormat NUMBER_FORMAT =
            NumberFormat.getIntegerInstance();
    private final static ResourceBundle BUNDLE =
            ResourceBundle.getBundle("ch/fhnw/jbackpack/Strings");
    private CardLayout cardLayout;
    private final DateFormat timeFormat;
    private final long start;
    private RdiffFileDatabase database;
    private Timer updateTimer = new Timer(1000, new ActionListener() {

        public void actionPerformed(ActionEvent e) {
            if (database == null) {
                cardLayout.show(cardPanel, "connectPanel");
            } else {
                switch (database.getSyncState()) {
                    case CHECKING:
                        // TODO: show something better
                        cardLayout.show(cardPanel, "connectPanel");
                        break;

                    case TRIMMING:
                        cardLayout.show(cardPanel, "trimmingPanel");
                        updateProgressBar(trimmingBackupProgressBar);
                        break;

                    case SYNCING:
                        cardLayout.show(cardPanel, "syncPanel");
                        updateProgressBar(syncBackupProgressBar);
                        fileTextField.setText(database.getCurrentFile());
                        fileTextField.setCaretPosition(0);
                        long directoryCounter = database.getDirectoryCounter();
                        directoryCounterTextField.setText(
                                NUMBER_FORMAT.format(directoryCounter));
                        long fileCounter = database.getFileCounter();
                        fileCounterTextField.setText(
                                NUMBER_FORMAT.format(fileCounter));
                        long sumCounter = directoryCounter + fileCounter;
                        sumCounterTextField.setText(
                                NUMBER_FORMAT.format(sumCounter));
                        break;

                    case COMPRESSING:
                        cardLayout.show(cardPanel, "compressPanel");
                }
            }
            long time = System.currentTimeMillis() - start;
            timeLabel.setText(timeFormat.format(time));
        }
    });

    /**
     * Creates new form DatabaseSyncDialog
     *
     * @param parent the parent frame
     */
    public DatabaseSyncDialog(Window parent) {
        super(parent, ModalityType.APPLICATION_MODAL);
        timeFormat = new SimpleDateFormat("HH:mm:ss");
        timeFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        initComponents();
        setLocationRelativeTo(parent);
        cardLayout = (CardLayout) cardPanel.getLayout();
        start = System.currentTimeMillis();
        updateTimer.setInitialDelay(0);
        updateTimer.start();
    }

    /**
     * sets the (already connected) RdiffFileDatabase
     *
     * @param database the database to set
     */
    public void setDatabase(RdiffFileDatabase database) {
        this.database = database;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        cardPanel = new javax.swing.JPanel();
        trimmingPanel = new javax.swing.JPanel();
        trimmingBackupLabel = new javax.swing.JLabel();
        trimmingBackupProgressBar = new javax.swing.JProgressBar();
        connectPanel = new javax.swing.JPanel();
        connectLabel = new javax.swing.JLabel();
        connectProgressBar = new javax.swing.JProgressBar();
        syncPanel = new javax.swing.JPanel();
        syncLabel = new javax.swing.JLabel();
        syncBackupLabel = new javax.swing.JLabel();
        syncBackupProgressBar = new javax.swing.JProgressBar();
        fileLabel = new javax.swing.JLabel();
        fileTextField = new javax.swing.JTextField();
        directoryCounterLabel = new javax.swing.JLabel();
        directoryCounterTextField = new javax.swing.JTextField();
        fileCounterLabel = new javax.swing.JLabel();
        fileCounterTextField = new javax.swing.JTextField();
        sumCounterLabel = new javax.swing.JLabel();
        sumCounterTextField = new javax.swing.JTextField();
        compressPanel = new javax.swing.JPanel();
        compressLabel = new javax.swing.JLabel();
        compressProgressBar = new javax.swing.JProgressBar();
        timeLabel = new javax.swing.JLabel();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("ch/fhnw/jbackpack/Strings"); // NOI18N
        setTitle(bundle.getString("DatabaseSyncDialog.title")); // NOI18N
        getContentPane().setLayout(new java.awt.GridBagLayout());

        cardPanel.setLayout(new java.awt.CardLayout());

        trimmingPanel.setLayout(new java.awt.GridBagLayout());

        trimmingBackupLabel.setText(bundle.getString("DatabaseSyncDialog.trimmingBackupLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        trimmingPanel.add(trimmingBackupLabel, gridBagConstraints);

        trimmingBackupProgressBar.setStringPainted(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 10, 0, 10);
        trimmingPanel.add(trimmingBackupProgressBar, gridBagConstraints);

        cardPanel.add(trimmingPanel, "trimmingPanel");

        connectPanel.setLayout(new java.awt.GridBagLayout());

        connectLabel.setText(bundle.getString("DatabaseSyncDialog.connectLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        connectPanel.add(connectLabel, gridBagConstraints);

        connectProgressBar.setIndeterminate(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        connectPanel.add(connectProgressBar, gridBagConstraints);

        cardPanel.add(connectPanel, "connectPanel");

        syncPanel.setLayout(new java.awt.GridBagLayout());

        syncLabel.setText(bundle.getString("DatabaseSyncDialog.syncLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.insets = new java.awt.Insets(15, 0, 0, 0);
        syncPanel.add(syncLabel, gridBagConstraints);

        syncBackupLabel.setText(bundle.getString("DatabaseSyncDialog.syncBackupLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 0, 0);
        syncPanel.add(syncBackupLabel, gridBagConstraints);

        syncBackupProgressBar.setStringPainted(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 0, 10);
        syncPanel.add(syncBackupProgressBar, gridBagConstraints);

        fileLabel.setText(bundle.getString("DatabaseSyncDialog.fileLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 10, 0, 0);
        syncPanel.add(fileLabel, gridBagConstraints);

        fileTextField.setColumns(40);
        fileTextField.setEditable(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 10);
        syncPanel.add(fileTextField, gridBagConstraints);

        directoryCounterLabel.setText(bundle.getString("DatabaseSyncDialog.directoryCounterLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 10, 0, 0);
        syncPanel.add(directoryCounterLabel, gridBagConstraints);

        directoryCounterTextField.setColumns(10);
        directoryCounterTextField.setEditable(false);
        directoryCounterTextField.setHorizontalAlignment(javax.swing.JTextField.TRAILING);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 10);
        syncPanel.add(directoryCounterTextField, gridBagConstraints);

        fileCounterLabel.setText(bundle.getString("DatabaseSyncDialog.fileCounterLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(2, 10, 0, 0);
        syncPanel.add(fileCounterLabel, gridBagConstraints);

        fileCounterTextField.setColumns(10);
        fileCounterTextField.setEditable(false);
        fileCounterTextField.setHorizontalAlignment(javax.swing.JTextField.TRAILING);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 5, 0, 10);
        syncPanel.add(fileCounterTextField, gridBagConstraints);

        sumCounterLabel.setText(bundle.getString("DatabaseSyncDialog.sumCounterLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(2, 10, 10, 0);
        syncPanel.add(sumCounterLabel, gridBagConstraints);

        sumCounterTextField.setColumns(10);
        sumCounterTextField.setEditable(false);
        sumCounterTextField.setHorizontalAlignment(javax.swing.JTextField.TRAILING);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 5, 10, 10);
        syncPanel.add(sumCounterTextField, gridBagConstraints);

        cardPanel.add(syncPanel, "syncPanel");

        compressPanel.setLayout(new java.awt.GridBagLayout());

        compressLabel.setText(bundle.getString("DatabaseSyncDialog.compressLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        compressPanel.add(compressLabel, gridBagConstraints);

        compressProgressBar.setIndeterminate(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        compressPanel.add(compressProgressBar, gridBagConstraints);

        cardPanel.add(compressPanel, "compressPanel");

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        getContentPane().add(cardPanel, gridBagConstraints);

        timeLabel.setText(bundle.getString("DatabaseSyncDialog.timeLabel.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(5, 19, 10, 0);
        getContentPane().add(timeLabel, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void updateProgressBar(JProgressBar progressbar) {
        int maxIncrementCounter = database.getMaxIncrementCounter();
        Date currentTimestamp = database.getCurrentTimestamp();
        if (currentTimestamp == null) {
            return;
        }
        String dateString = DATE_FORMAT.format(currentTimestamp);
        if (maxIncrementCounter == 0) {
            progressbar.setString(dateString);
        } else {
            int incrementCounter = database.getIncrementCounter();
            String string = BUNDLE.getString("SyncProgress");
            string = MessageFormat.format(string,
                    dateString, incrementCounter, maxIncrementCounter);
            progressbar.setString(string);
            int progress = (incrementCounter * 100) / maxIncrementCounter;
            progressbar.setValue(progress);
        }
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel cardPanel;
    private javax.swing.JLabel compressLabel;
    private javax.swing.JPanel compressPanel;
    private javax.swing.JProgressBar compressProgressBar;
    private javax.swing.JLabel connectLabel;
    private javax.swing.JPanel connectPanel;
    private javax.swing.JProgressBar connectProgressBar;
    private javax.swing.JLabel directoryCounterLabel;
    private javax.swing.JTextField directoryCounterTextField;
    private javax.swing.JLabel fileCounterLabel;
    private javax.swing.JTextField fileCounterTextField;
    private javax.swing.JLabel fileLabel;
    private javax.swing.JTextField fileTextField;
    private javax.swing.JLabel sumCounterLabel;
    private javax.swing.JTextField sumCounterTextField;
    private javax.swing.JLabel syncBackupLabel;
    private javax.swing.JProgressBar syncBackupProgressBar;
    private javax.swing.JLabel syncLabel;
    private javax.swing.JPanel syncPanel;
    private javax.swing.JLabel timeLabel;
    private javax.swing.JLabel trimmingBackupLabel;
    private javax.swing.JProgressBar trimmingBackupProgressBar;
    private javax.swing.JPanel trimmingPanel;
    // End of variables declaration//GEN-END:variables
}
