/*
  searchCmnds.c, written by Rhett "Jonzy" Jones 

  Jonzy's Universal Gopher Hierarchy Excavation And Display.
  Excavates through gopher menus and displays the hierarchy
  of the menus encountered

  Copyright (C) 1993, 1994 University of Utah Computer Center.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program (look for the file called COPYING);
  if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              */

/*
 * Description:	This module takes care of some special commands for use
 *	by the jugtail search engine.  A special command is defined
 *	as one of the following:
 *		?all what
 *		?datafile
 *		?datasize
 *		?help [what]
 *		?limit=n what
 *		?version [what]
 *		?range=start-stop what
 *		?pathway what		NOT supported yet.
 *	where 'what' is a standard search string, anything enclosed
 *	in square brackets is optional, and all special commands must
 *	be preceeded with '?'.
 *
 *	The pathway command is currently not supported.
 *
 *	WARNING:  Any attempt to alter the words for any of these
 *	commands will result in a jugtail that is not compatible with
 *	future versions of jugtail.  These special commands are
 *	inherent to the jugtail protocol.
 *
 */

#include "stdinc.h"

static short StillNeed2Parse (char *string, short *theCommand,
			      long *limit, long *rangeStart, long *rangeEnd);
static long GetDateFileSize (char *fName);
static int SendDataFile (char *fName);
static int HandleSpecialCommand (char *string, short theCommand);
char *SpecialCommand (char *what2find, long *limit,
		      long *rangeStart, long *rangeEnd);


#define RANGEERROR	-4	/* Error with range usage. */
#define PATHWAYERROR	-3	/* Error with pathway usage. */
#define CMDUSEERROR	-2	/* Generic usage error. */
#define LIMITERROR	-1	/* Error with limit usage. */

/* Various jugtail commands followed by syntax. */
#define ALLCMND		1	/* ?all what */
#define DATACMND	2	/* ?datafile */
#define DATSZCMND	3	/* ?datasize */
#define HELPCMND	4	/* ?help [what] */
#define LIMITCMND	5	/* ?limit=n what */
#define VERSIONCMND	6	/* ?version [what] */
#define RANGECMND	7	/* ?range=start-end what */
#ifdef DO_PATHWAY
#define PATHWAYCMND	8	/* ?pathway what */
#endif

/* This variable lists the special commands for use by the jugtail search engine. */
static char *specialCmnds[] = { "all",	/* Return all entries in excess of 1024. */
  "datafile",			/* Return the datafile being used. */
  "datasize",			/* The size of the datafile. */
  "help",			/* Return a link to a help file. */
  "limit",			/* Return at most the first n entries. */
  "version",			/* Return the version of jugtail. */
  "range",			/* Return a range of items. */
#ifdef DO_PATHWAY
  "pathway",			/* Return the pathway for a selection. */
#endif
  0
};

#ifdef DO_PATHWAY
char *pathString;		/* The selStr to find the path of. */
#endif

extern char *fileName,		/* Declared in "jugtailConf.c. */
 *veronica;			/* Declared in "jugtailConf.c". */
extern char *jugtailhelp,	/* Declared in "searchCmnds.c". */
 *usagerror, *hostname, *jugtailhelptitl;
extern int port2use;		/* Declared in "searchCmnds.c". */

extern int SendBuffer ();	/* Defined  in "tree.c". */
extern int SendString ();	/* Defined  in "tree.c". */

/*****************************************************************************
 * StillNeed2Parse determines if we have a special command and if so assigns
 * the appropriate command to 'theCommand', which includes any command errors
 * encountered.  This routine returns true if there is still information to
 * parse, and false otherwise.
 ****************************************************************************/
static short
StillNeed2Parse (char *string, short *theCommand, 
		 long *limit, long *rangeStart, long *rangeEnd)
     /* string: The line of text given.
	theCommand: Do we have a special command?
	limit: The number of items to return.
	rangeStart: The start of a range.
	rangeEnd: The end of a range.             */
{
  char *str,			/* Position in 'string'. */
    **s,				/* The special commands. */
    theSpecialCmnd[12];		/* The special command the user issued. */
  short i;			/* A loop counter. */

  /* Initialize some variables. */
  *rangeStart = *rangeEnd = *theCommand = 0;
  theSpecialCmnd[0] = '\0';

  /* Skip any whitespace. */
  for (str = string; *str && isspace (*str); str++);
  
  if (*str == '?')		/* Looks like a special command. */
    {
      /* Get a copy of the special command. */
      if (!strcmp (str, "?"))	/* Treat it like ?help. */
	strcpy (theSpecialCmnd, "help");
      else
	{
	  for (++str, i = 0; i < 11 && isalpha (*str); i++, str++)
	    theSpecialCmnd[i] = *str;
	  theSpecialCmnd[i] = '\0';
	}
      
      /* Locate the special command we found. */
      for (*theCommand = 1, s = specialCmnds; *s; s++, ++*theCommand)
	if (!strcmp (*s, theSpecialCmnd))
	  break;

      switch (*theCommand)
	{
	case ALLCMND:
	  *limit = -1;
	  break;
	case DATACMND:
	case DATSZCMND:
	case HELPCMND:
	  break;
	case LIMITCMND:
	  *limit = 0;
	  if (*str == '=' && isdigit (*++str))
	    for (*limit = 0; isdigit (*str); str++)
	      *limit = *limit * 10 + *str - '0';
	  else
	    *theCommand = LIMITERROR;
	  break;
	case VERSIONCMND:
	  break;
	case RANGECMND:
	  if (*str == '=' && isdigit (*++str))
	    {
	      for (*rangeStart = 0; isdigit (*str); str++)
		*rangeStart = *rangeStart * 10 + *str - '0';
	      if (*str == '-')
		for (++str, *rangeEnd = 0; isdigit (*str); str++)
		  *rangeEnd = *rangeEnd * 10 + *str - '0';
	      else
		{
		  *rangeStart = *rangeEnd = 0;
		  *theCommand = RANGEERROR;
		}
	    }
	  else
	    *theCommand = RANGEERROR;
	  break;
#ifdef DO_PATHWAY
	case PATHWAYCMND:
	  if (!*str)
	    {
	      *theCommand = PATHWAYERROR;
	      break;
	    }
	  if (*str)		/* Skip any whitespace. */
	    for (++str; isspace (*str); str++);

	  /* Extract the pathString. */
	  if (*str == '"')
	    {
	      for (pathString = ++str; *str && *str != '"'; str++);
	      if (*str == '"')
		*str++ = '\0';
	      else
		{
		  *theCommand = PATHWAYERROR;
		  break;
		}
	    }
	  else
	    {
	      for (pathString = str; *str && !isspace (*str); str++);
	      *str = '\0';
	    }
	  break;
#endif
	default:		/* Oops we're in deep shit. */
	  *theCommand = CMDUSEERROR;
	  theSpecialCmnd[0] = '\0';
	  break;
	}

      /* Shft the special command out from 'string'. */
      if (*str)
	for (++str; isspace (*str); str++);	/* Skip any whitespace. */
      for (i = 0; *str; i++, str++)
	string[i] = *str;
      string[i] = '\0';
    }

  return (*string);

}				/* StillNeed2Parse */

/*****************************************************************************
 * GetDateFileSize returns the number of bytes in the file 'fname', and -1
 * if an error occured.
 ****************************************************************************/
static long
GetDateFileSize (char *fName)
     /* fName: Name of the datafile. */
{
  FILE *fp;			/* Pointer to the file. */
  long bytes;			/* Number of bytes in 'fName'. */
  char ptr[BUFSIZ];		/* Buffer with the file contents. */
  int itemsRead;		/* The number of bytes read. */

  if ((fp = fopen (fName, "r")))
    {
      for (bytes = 0; (itemsRead = fread (ptr, 1, BUFSIZ, fp));
	   bytes += itemsRead);
      fclose (fp);
      return (bytes);
    }
  else
    return (-1);

}				/* GetDateFileSize */


/*****************************************************************************
 * SendDataFile sends the datafile 'fName' to the client we are talking to.
 * This routine returns true is we had an error, and false otherwise.
 ****************************************************************************/
static int
SendDataFile (char *fName)
{
  FILE *fp;			/* Pointer to the file. */
  char ptr[BUFSIZ];		/* Buffer with the file contents. */
  int itemsRead;		/* The number of bytes read. */

  if ((fp = fopen (fName, "r")))
    {
      while (1)
	{
	  itemsRead = fread (ptr, 1, BUFSIZ, fp);
	  if (itemsRead)
	    {
	      if (SendBuffer (ptr, itemsRead) != itemsRead)
		break;
	    }
	  else
	    break;
	}
      fclose (fp);
      return (0);
    }
  else
    return (1);

}				/* SendDataFile */

/*****************************************************************************
 * WhichFile returns the pathway to the 'veronica' file "data.veronica" if
 * it exists, otherwise it returns the name of the data file 'fileName'.
 ****************************************************************************/
static char *
WhichFile (void)
{
  FILE *fp;

  if ((fp = fopen (veronica, "r")))
    {
      fclose (fp);
      return (veronica);
    }
  else
    return (fileName);

}				/* WhichFile */

/*****************************************************************************
 * HandleSpecialCommand returns true if 'theCommand' is an error or we
 * discover an error, and send the information back to the client following
 * gopher protocol.  Otherwise this routine returns false and sets some
 * variables or accomplishes the special command.
 ****************************************************************************/
static int
HandleSpecialCommand (char *string, short theCommand)
     /* string: The remainder of the input line.
	theCommand: The special command or error. */
{
  int error = 0;		/* Did we get an error? */
  char help[1024];		/* Line of text adhering to gopher protocol. */

  switch (theCommand)
    {
    case RANGEERROR:
      sprintf (help, "0range %s%s", usagerror, jugtailhelp);
      error = SendString (help);
      break;
    case PATHWAYERROR:
      sprintf (help, "0pathway %s%s", usagerror, jugtailhelp);
      error = SendString (help);
      break;
    case CMDUSEERROR:
      sprintf (help, "0jugtail %s%s", usagerror, jugtailhelp);
      error = SendString (help);
      break;
    case LIMITERROR:
      sprintf (help, "0limit %s%s", usagerror, jugtailhelp);
      error = SendString (help);
      break;
    case ALLCMND:
      if (!*string)
	{
	  sprintf (help, "0all %s%s", usagerror, jugtailhelp);
	  error = SendString (help);
	}
      break;
    case DATACMND:
      error = SendDataFile (WhichFile ());
      break;
    case DATSZCMND:
      sprintf (help, "0%ld bytes\t?datafile\t%s\t%d",
	       GetDateFileSize (WhichFile ()), hostname, port2use);
      error = SendString (help);
      break;
    case HELPCMND:
      sprintf (help, "0%s%s", jugtailhelptitl, jugtailhelp);
      SendString (help);
      break;
    case LIMITCMND:
      if (!*string)
	{
	  sprintf (help, "0limit usage error%s", jugtailhelp);
	  error = SendString (help);
	}
      break;
    case VERSIONCMND:
      sprintf (help, "0This version of jugtail is %s%s", VERSION,
	       jugtailhelp);
      SendString (help);
      break;
    case RANGECMND:
      break;
#ifdef DO_PATHWAY
    case PATHWAYCMND:
      /* exec pathway on pathString. */
      break;
#endif
    }
  return (error);

}				/* HandleSpecialCommand */

/*****************************************************************************
 * SpecialCommand handles any special command the jugtail search engine
 * might get, and returns nil if there is nothing to parse out, and the
 * string to continue parsing if there is any.
 ****************************************************************************/
char *
SpecialCommand (char *what2find, long *limit, 
		long *rangeStart, long *rangeEnd)
     /* what2find: The input line.
	limit: The maximum number of items to return.
	rangeStart: The start of a range.
	rangeEnd: The end of a range. */
{
  short theCommand;		/* Do we have a special command? */

  if (StillNeed2Parse (what2find, &theCommand, limit, rangeStart, rangeEnd))
    {
      if (theCommand)
	if (HandleSpecialCommand (what2find, theCommand))
	  return ((char *) NULL);
    }
  else if (theCommand)
    {
      HandleSpecialCommand (what2find, theCommand);
      return ((char *) NULL);
    }

  return (what2find);

}				/* SpecialCommand */
