#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import playerbase as PlayerBase
import playerthread as PlayerThread
import mediasong as MediaSong


class MPlayerPlayer(PlayerBase.PlayerBase):
    """ A Non-Blocking player for MPlayer """

    # The PlayerThread that runs MPlayer.
    __mplayer = None

    def __init__(self, volumeLevel, additionalArguments=""):
        """ Default constructor

         Initialises and starts the player thread for using MPlayer.
         int volumeLevel: is the percentage level for play music.
         string additionalArguments: are any additional arguments to pass along
           to MPlayer when playing a song.
        """
        PlayerBase.__init__("MPlayer")
        self.__mplayer = PlayerThread.PlayerThread("mplayer", \
            "-slave -quiet " + additionalArguments, volumeLevel)
        # Setup the signals
        self.__mplayer.signalPlay = ""
        self.__mplayer.signalPause = "pause"
        self.__mplayer.signalUnpause = "pause"
        self.__mplayer.signalStop = "quit"
        self.__mplayer.signalVolume = "volume"
        self.__mplayer.signalTotalTime = "get_time_length"
        self.__mplayer.signalPositionTime = "get_time_pos"
        self.__mplayer.signalSetPositionTime = "seek"
        self.__mplayer.signalQuit = "quit"

    def __del__(self):
        """ Ensure the player thread has stopped. """
        if self.isPlaying() or self.isPaused or self.isStarting:
            self.stopSong()
        if self.isRunning:
            self.quitRunning()
            self.__mplayer.join(0.5)

    def quitRunning(self):
        """ Signal MPlayer to quit running. """
        self.__mplayer.quitRunning()

    def isSongEnded(self):
        """ Determines if a song has ended.

         return boolean: true if the song has finished, false else-wise
        """
        return self.__mplayer.isSongEnded()

    def playSong(self, song, volume):
        """ Starts to play a song.

         Hands the song to play off to MPlayer.
         MediaSong song: is the song to play
         int volume: is the volume percentage level.
         return boolean: true if the signal to play succeeds, false
           if one of the pre-conditions for starting to play is not met at the
           time of sending the signal.
        """
        print "MPlayerPlayer.playSong(self, song, volume)", song, volume
        if song.filePath is not None:
            return self.__mplayer.play(song.filePath, volume)
        else:
            return False

    def isRunning(self):
        """ Determines if MPlayer is running.

         return boolean: true if the player is running, false else-wise
        """
        return self.__mplayer.isRunning()

    def isStarting(self):
        """ Determines if the player is in the process of starting up.

         return boolean: true if the player is starting up, false else-wise
        """
        return self.__mplayer.isStarting()

    def isPlaying(self):
        """ Indicates if the player is playing a song.

         return boolean: true if a song is playing, false else-wise
        """
        return self.__mplayer.isPlaying()

    def isPaused(self):
        """ Indicates if the player is paused

         return boolean: true if a song is paused, false else-wise
        """
        return self.__mplayer.isPaused()

    def stopSong(self):
        """ Signal to stop the currently playing song. """
        self.__mplayer.stop()

    def pauseSong(self):
        """ Signal to pause the currently playing song. """
        self.__mplayer.pause()

    def unpauseSong(self):
        """ Signal to un-pause the currently playing song. """
        self.__mplayer.unpause()

    def setVolumeLevel(self, volumeLevel):
        """ Signal to set the volume level of MPlayer.

         Signal to change the volume level of MPlayer.
         int volumeLevel: is the percentage level to set the volume at.
         return boolean: True if the signal is successfully sent, false
           if one of the pre-conditions for setting the volume level is not
           met at the time of sending the signal.
        """
        return self.__mplayer.setVolumeLevel(volumeLevel)

    def volumeLevel(self):
        """ Current volume level of MPlayer.
         return int: the percentage level of volume of MPlayer.
        """
        return self.__mplayer.volumeLevel()

    def setPlayPosition(self, seconds):
        """ Signal to change the position of the currently playing song.

         float seconds: is the position in the song to play from in terms of
           the number of seconds starting from the beginning of the song.
         return boolean: True if the signal is successfully sent, false
           if one of the pre-conditions for setting the position is not met at
           the time of sending the the signal.
        """
        return self.__mplayer.setPlayPosition(seconds)

    def positionInSong(self):
        """ Accessor to the current position in a song.

         return float: the total seconds that have passed in the currently
           playing song.
        """
        return self.__mplayer.positionInSong()

    def totalSongTime(self):
        """ Accessor to the total time in a song.

         return float: the total number of seconds in the current song.
        """
        return self.__mplayer.totalSongTime()
