#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import sys
import ConfigParser

import wx

import medialibrarydialogue as MediaLibraryDialogue
import kapdialogue as KapDialogue
import searchwindow as SearchWindow
import playlistwindow as PlayListWindow
import playerwindow as PlayerWindow
import medialibrarythread as MediaLibraryThread
import playerfactory as PlayerFactory
import mediasong as MediaSong

# The current version of Kap
_VERSION = "0.3.0"


class Kap(wx.Frame):
    """ The programme interface for running a KJ show. """

    """
     ATTRIBUTES
    """
    # The settings to use for the programme
    __sfCnfgPrsrKap = ConfigParser.SafeConfigParser()
    # The collection of all the audio and video media
    __database = MediaLibraryThread.MediaLibraryThread()
    # The default size of the window for Kap
    __windowSize = wx.Size(800, 600)
    # The minimum size for Kap that's still usable.
    __MINIMUM_SIZE = wx.Size(480, 320)
    # The update timer for the track position display
    __trackTimer = None
    # The refresh rate for the track position display, in milliseconds
    __trackRefreshRate = 100
    # The update timer for the search result display
    __searchTimer = None
    # The refresh rate for the search result display, in milliseconds
    __searchRefreshRate = 500
    # Indicates the number of fields in the status bar
    __STATUS_FIELD_COUNT = 2
    # The field for general programme status information
    __GENERAL_STATUS_FIELD = 0
    # The field for player status information
    __PLAYER_STATUS_FIELD = 1
    # The media player factory for Kap
    __playerFactory = None
    # The volume level for the DJ player
    __volumeLevel = 50
    # Indicates the programme should play songs continuously from the top of
    #    the play-list.
    __playContinuous = False
    # Indicates the programme should randomly play a song from the play-list.
    __playRandom = False
    # Indicates the current song should be removed from the play-list
    __removeFromPlayList = True
    # Additional arguments that should be passed on the the player.
    __additionalPlayerArguements = ""
    # Holds the current playing song
    __currentSong = None

    """
     EVENT ID
    """
    # Used to signal to edit the media library.
    __ID_LIBRARY = wx.NewId()
    # Used to signal to edit Kap settings.
    __ID_KAP = wx.NewId()
    # Used to signal to load a media file into the play-list.
    __ID_LOAD = wx.NewId()
    # Used to signal to exit the programme
    __ID_EXIT = wx.NewId()
    # Used to signal to adjust the volume level.
    __ID_VOLUME = wx.NewId()
    # Used to signal to play a song from the play-list.
    __ID_PLAY = wx.NewId()
    # Used to signal to stop the currently playing song
    __ID_STOP = wx.NewId()
    # Used to signal to pause the currently playing song.
    __ID_PAUSE = wx.NewId()
    # Used to signal to move the position of a song in the play-list
    __ID_MOVE = wx.NewId()
    # Used to signal to display the About dialogue.
    __ID_ABOUT = wx.NewId()
    # Used to signal to update the DJ player information.
    __ID_TRACK_TIMER = wx.NewId()
    # Used to signal to update the search result information.
    __ID_SEARCH_TIMER = wx.NewId()
    # Used to signal to update the play continuously option.
    __ID_CONTINUOUS = wx.NewId()
    # Used to signal to update the play randomly option.
    __ID_RANDOM = wx.NewId()

    def __init__(self, parent, id, configuration):
        """ Kap application widget constructor

         wx.Window parent: is the parent window.
         wx.WindowID id: is the unique identification number for the window
         SafeConfigParser configuration: is the settings for Kap.
        """
        wx.Frame.__init__(self, parent, id, 'Kap')

        self.__sfCnfgPrsrKap = configuration
        self.__playerFactory = PlayerFactory.PlayerFactory()
        self.__readSettings()
        self.__createMenus()
        self.__createControls()
        self.__bindEvents()
        self.__doLayout()
        self.__trackTimer.Start(self.__trackRefreshRate)

    def __readSettings(self):
        """ Read the settings and use them.

         All class options are set from their defaults to the values in the
           configuration file.
         Creates a temporary holder for the audio and video search information
           for the creation of the SearchWindow
        """
        self.__database.readSettings(self.__sfCnfgPrsrKap)
        if self.__sfCnfgPrsrKap.has_section('MediaLibrary'):
            if self.__sfCnfgPrsrKap.has_option('MediaLibrary', \
                    "searchAudios"):
                self.__audio = self.__sfCnfgPrsrKap.getboolean( \
                    "MediaLibrary", "searchAudios")
            if self.__sfCnfgPrsrKap.has_option('MediaLibrary', \
                    "searchVideos"):
                self.__video = self.__sfCnfgPrsrKap.getboolean( \
                    "MediaLibrary", "searchVideos")
        else:
            self.__audio = True
            self.__video = True
        if self.__sfCnfgPrsrKap.has_section('Kap'):
            if self.__sfCnfgPrsrKap.has_option('Kap', "windowSize"):
                 size = self.__sfCnfgPrsrKap.get("Kap", "windowSize")
                 # Convert string size into width and height integers
                 width = size.replace("(", "").replace(")", "")
                 width = int(width.split(",")[0])
                 height = size.replace("(", "").replace(")", "")
                 height = int(height.split(",")[1])
                 self.__windowSize = wx.Size(width, height)
            if self.__sfCnfgPrsrKap.has_option('Kap', "searcRefreshRate"):
                self.__searchRefreshRate = \
                    self.__sfCnfgPrsrKap.getint("Kap", "searcRefreshRate")
            if self.__sfCnfgPrsrKap.has_option('Kap', "removeFromPlayList"):
                self.__removeFromPlayList = \
                    self.__sfCnfgPrsrKap.getboolean("Kap", "removeFromPlayList")
        if self.__sfCnfgPrsrKap.has_section('Player'):
            if self.__sfCnfgPrsrKap.has_option('Player', \
                    "additionalPlayerArguements"):
                self.__additionalPlayerArguements = \
                    self.__sfCnfgPrsrKap.get("Player", \
                    "additionalPlayerArguements")
            if self.__sfCnfgPrsrKap.has_option('Player', "volumeLevel"):
                self.__volumeLevel = self.__sfCnfgPrsrKap.getint("Player", \
                    "volumeLevel")
            if self.__sfCnfgPrsrKap.has_option('Player', "playerType"):
                self.__playerFactory.type_ = self.__sfCnfgPrsrKap.get( \
                    "Player", "playerType")
            if self.__sfCnfgPrsrKap.has_option('Player', "trackRefreshRate"):
                self.__trackRefreshRate = \
                    self.__sfCnfgPrsrKap.getint("Player", "trackRefreshRate")
            if self.__sfCnfgPrsrKap.has_option('Player', "playContinuous"):
                self.__playContinuous = \
                    self.__sfCnfgPrsrKap.getboolean("Player", "playContinuous")
            if self.__sfCnfgPrsrKap.has_option('Player', "playRandom"):
                self.__playRandom = \
                    self.__sfCnfgPrsrKap.getboolean("Player", "playRandom")

    def __writeSettings(self):
        """ Save the current settings.

         All the class options are entered into the configuration object. Sets
           the media library, player, and programme options.
        """
        self.StatusBar.SetStatusText("Saving media library...", \
            self.__GENERAL_STATUS_FIELD)
        self.__database.writeSettings(self.__sfCnfgPrsrKap)
        if not self.__sfCnfgPrsrKap.has_section('MediaLibrary'):
            self.__sfCnfgPrsrKap.add_section('MediaLibrary')
        self.__sfCnfgPrsrKap.set('MediaLibrary', 'searchAudios', \
            "%s" % self.__searchWindow.searchForAudios())
        self.__sfCnfgPrsrKap.set('MediaLibrary', 'searchVideos', \
            "%s" % self.__searchWindow.searchForVideos())
        # Write Player Settings
        self.StatusBar.SetStatusText("Saving player settings...", \
            self.__GENERAL_STATUS_FIELD)
        if not self.__sfCnfgPrsrKap.has_section('Player'):
            self.__sfCnfgPrsrKap.add_section('Player')
        self.__sfCnfgPrsrKap.set('Player', 'additionalPlayerArguements', \
            "%s" % self.__additionalPlayerArguements)
        self.__sfCnfgPrsrKap.set('Player', 'volumeLevel', "%s" % \
            self.__volumeLevel)
        self.__sfCnfgPrsrKap.set('Player', 'playerType', "%s" % \
            self.__playerFactory.type_)
        self.__sfCnfgPrsrKap.set('Player', 'trackRefreshRate', "%s" % \
            self.__trackRefreshRate)
        self.__sfCnfgPrsrKap.set('Player', 'playContinuous', "%s" % \
            self.__playContinuous)
        self.__sfCnfgPrsrKap.set('Player', 'playRandom', "%s" % \
            self.__playRandom)
        # Write Kap Settings
        self.StatusBar.SetStatusText("Saving application settings...", self.__GENERAL_STATUS_FIELD)
        if not self.__sfCnfgPrsrKap.has_section('Kap'):
            self.__sfCnfgPrsrKap.add_section('Kap')
        self.__sfCnfgPrsrKap.set('Kap', 'windowSize', "%s" % self.GetSize())
        self.__sfCnfgPrsrKap.set('Kap', 'searchRefreshRate', "%s" % \
            self.__searchRefreshRate)
        self.__sfCnfgPrsrKap.set('Kap', 'removeFromPlayList', "%s" % \
            self.__removeFromPlayList)

    def __createMenus(self):
        """ Create the file menu options.

         Creates the File, Player, Settings, and Help menus.
        """
        # Create file menu
        self.__mnFile = wx.Menu()
        self.__mnFile.Append(self.__ID_LOAD, "&Load Media", \
            "Load a media files directly into the play-list")
        self.__mnFile.AppendSeparator()
        self.__mnFile.Append(self.__ID_EXIT, "E&xit", "Exit application")
        # Create player menu
        self.__mnPlayer = wx.Menu()
        self.__mnPlayer.AppendCheckItem(self.__ID_CONTINUOUS,\
            "&Continuous Play", "Play continuously from the play-list")
        self.__mnPlayer.Check(self.__ID_CONTINUOUS, self.__playContinuous)
        self.__mnPlayer.AppendCheckItem(self.__ID_RANDOM,"&Random Play", \
            "Play random music from the play-list")
        self.__mnPlayer.Check(self.__ID_RANDOM, self.__playRandom)
        # Create settings menu
        self.__mnSettings = wx.Menu()
        self.__mnSettings.Append(self.__ID_LIBRARY, \
            "Configure &Media Library", \
            "Configure audio and video media library")
        self.__mnSettings.Append(self.__ID_KAP, "Configure &Kap", \
            "Configure Kap settings")
        # Create help menu
        self.__mnHelp = wx.Menu()
        self.__mnHelp.Append(self.__ID_ABOUT, "&About", \
            "Application information")

    def __createControls(self):
        """ Create the user interface

         Creates each of the windows for the user interface. The temporary
           holders for the creation of SearchWindow are removed at this point.
        """
        self.__playerFactory.create(self.__volumeLevel, \
            self.__additionalPlayerArguements)
        self.__playerWindow = PlayerWindow.PlayerWindow(self, \
            volume=self.__volumeLevel)
        self.__searchTimer = wx.Timer(self, self.__ID_SEARCH_TIMER)

        # Create Search Window And Play List
        self.__splttrWndwSearchPlay = wx.SplitterWindow(self, wx.NewId())
        self.__splttrWndwSearchPlay.SetMinimumPaneSize( \
            self.__MINIMUM_SIZE.GetWidth() / 4)
        self.__splttrWndwSearchPlay.SetSashGravity(0.85)
        self.__searchWindow = SearchWindow.SearchWindow( \
            self.__splttrWndwSearchPlay, self.__audio, self.__video)
        del self.__audio
        del self.__video
        self.__playListWindow = PlayListWindow.PlayListWindow( \
            self.__splttrWndwSearchPlay)
        self.__trackTimer = wx.Timer(self, self.__ID_TRACK_TIMER)

        # Create the status bar display
        self.CreateStatusBar()
        self.StatusBar.SetFieldsCount(self.__STATUS_FIELD_COUNT)
        self.StatusBar.SetStatusText("Search Ready", \
            self.__GENERAL_STATUS_FIELD)
        self.StatusBar.SetStatusText("Player Ready", \
            self.__PLAYER_STATUS_FIELD)

    def __bindEvents(self):
        """ Connects all the needed events.

         Connects each of the control objects created to their corresponding
           methods.
        """
        self.Bind(wx.EVT_TIMER, self.__updateTrackPosition, \
            self.__trackTimer)
        self.Bind(wx.EVT_TIMER, self.__updateSearchResults, \
            self.__searchTimer)
        # Menu Events
        wx.EVT_MENU(self, self.__ID_LIBRARY, self.__onMediaLibrary)
        wx.EVT_MENU(self, self.__ID_KAP, self.__onConfigureKap)
        wx.EVT_MENU(self, self.__ID_EXIT, self.__onExit)
        wx.EVT_MENU(self, self.__ID_LOAD, self.__onLoadFile)
        wx.EVT_MENU(self, self.__ID_ABOUT, self.__onAbout)
        wx.EVT_MENU(self, self.__ID_CONTINUOUS, self.__onMenuPlayer)
        wx.EVT_MENU(self, self.__ID_RANDOM, self.__onMenuPlayer)
        wx.EVT_CLOSE(self, self.__onExit)
        # Search Window Events
        wx.EVT_TEXT_ENTER(self, self.__searchWindow.ID_SEARCH_ENTRY, \
            self.__onFindSong)
        wx.EVT_SEARCHCTRL_CANCEL_BTN(self, \
            self.__searchWindow.ID_SEARCH_ENTRY, self.__onCancelFindSong)
        wx.EVT_LIST_ITEM_ACTIVATED(self, \
            self.__searchWindow.ID_SEARCH_RESULTS, \
            self.__onAddSongToPlayList)
        # Player Window Events
        wx.EVT_BUTTON(self, self.__playerWindow.ID_PLAY, self.__onPlaySong)
        wx.EVT_BUTTON(self, self.__playerWindow.ID_RESTART, \
            self.__onRestartSong)
        wx.EVT_BUTTON(self, self.__playerWindow.ID_STOP, self.__onStopSong)
        wx.EVT_COMMAND_SCROLL(self, self.__playerWindow.ID_VOLUME, \
            self.__onChangeVolume)
        wx.EVT_COMMAND_SCROLL(self, self.__playerWindow.ID_TRACK_POSITION, \
            self.__onChangeTrackPosition)

    def __doLayout(self):
        """ Creates a visually pleasing look for the controls.

         All the control objects are placed on the window.
        """
        # Add the menu bar
        self.__mnBrMain = wx.MenuBar()
        self.__mnBrMain.Append(self.__mnFile,"&File")
        self.__mnBrMain.Append(self.__mnPlayer,"&Player")
        self.__mnBrMain.Append(self.__mnSettings,"&Settings")
        self.__mnBrMain.Append(self.__mnHelp,"&Help")
        self.SetMenuBar(self.__mnBrMain)

        # Create a split window layout for a Search Window and a Play-List
        self.__splttrWndwSearchPlay.SplitVertically(self.__searchWindow, \
            self.__playListWindow)

        # Create the main layout
        self.__bxSzrMainLayout = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrMainLayout.Add(self.__playerWindow, 0, wx.ALL | wx.EXPAND)
        self.__bxSzrMainLayout.AddSpacer(5)
        self.__bxSzrMainLayout.Add(self.__splttrWndwSearchPlay, 1, wx.ALL | \
            wx.EXPAND)
        self.SetAutoLayout(True)
        self.SetSizer(self.__bxSzrMainLayout)
        self.SetMinSize(self.__MINIMUM_SIZE)
        self.SetSize(self.__windowSize)

    def __onMediaLibrary(self, event):
        """ Interface for the media library database

         Allows a user to adjust settings for the media library, and to update
           the media library.
         wx.Event event: is the event object (Not Used)
        """
        mli = MediaLibraryDialogue.MediaLibraryDialogue(self, \
            self.__database.searchPaths())
        mli.ShowModal()
        if mli.blSaveSettings:
            self.StatusBar.SetStatusText("Updating Media Library...", \
                self.__GENERAL_STATUS_FIELD)
            songsInLibrary = self.__database.buildLibrary( \
                mli.mediaLibraryPaths())
            wx.MessageBox("Library contains %d songs now." % \
                songsInLibrary, "Media Library", wx.OK | \
                wx.ICON_INFORMATION | wx.STAY_ON_TOP)
            self.StatusBar.SetStatusText("Media Library Updated", \
                self.__GENERAL_STATUS_FIELD)

    def __onConfigureKap(self, event):
        """ Interface for configuring Kap.

         Allows a user to adjust settings for Kap.
         wx.Event event: is the event object (Not Used)
        """
        kd = KapDialogue.KapDialogue(self, \
            self.__additionalPlayerArguements, self.__trackRefreshRate, \
                self.__searchRefreshRate, self.__removeFromPlayList)
        kd.ShowModal()
        if kd.blSaveSettings:
            # Update the player arguments
            self.__additionalPlayerArguements = kd.playerArguments()
            self.__playerFactory.destroy()
            self.__playerFactory.create(self.__volumeLevel, \
                self.__additionalPlayerArguements)
            self.__handleEndSong()

            # Update the player window refresh rate
            self.__trackRefreshRate = kd.playerRefreshRate()
            self.__trackTimer.Stop()
            self.__trackTimer.Start(self.__trackRefreshRate)
            self.__searchRefreshRate = kd.searchRefreshRate()
            self.__removeFromPlayList = kd.removeFromPlayList()

    def __onFindSong(self, event):
        """ Starts the database searching for songs.

         The database looks for songs that match the criteria in the search
           window.
         wx.Event event: is the event object (Not Used)
        """
        if self.__searchTimer.IsRunning():
            self.__cancelSearch()

        self.StatusBar.SetStatusText("Searching For Media...", \
            self.__GENERAL_STATUS_FIELD)
        self.__database.findSongs( \
            self.__searchWindow.searchTerms(), \
            self.__searchWindow.searchForAudios(), \
            self.__searchWindow.searchForVideos())
        self.__searchTimer.Start(self.__searchRefreshRate)

    def __onCancelFindSong(self, event):
        """ Stops a user library search.

         wx.Event event: is the event object (Not Used)
        """
        self.__searchWindow.clearSearchTerms()
        self.__cancelSearch()

    def __cancelSearch(self):
        """ Stops a user library search. """
        self.__searchTimer.Stop()
        self.StatusBar.SetStatusText("Search Cancelled", \
            self.__GENERAL_STATUS_FIELD)

    def __updateSearchResults(self, event):
        """ Updates the search results window.

         wx.Event event: is the event object (Not Used)
        """
        if self.__database.searchFinished():
            self.__searchWindow.clearSearchResults()
            songs = self.__database.songsFound()
            for song in songs:
                if self.__searchTimer.IsRunning():
                    self.StatusBar.SetStatusText("Found %s - %s..." % \
                        (song.title, song.artist), \
                        self.__GENERAL_STATUS_FIELD)
                    self.__searchWindow.addSearchResult(song)
                else:
                    break
                wx.SafeYield(self, True)
            self.StatusBar.SetStatusText("Search Completed", \
                self.__GENERAL_STATUS_FIELD)
            if self.__searchTimer.IsRunning():
                self.__searchTimer.Stop()

    def __onLoadFile(self, event):
        """ An interface to load songs directly into the play-list.

         A file dialogue for the user to select songs to be placed into the
           play-list.
         wx.Event event: is the event object (Not Used)
        """
        mediaFilter = ''
        for media in self.__database.supportedMedia():
            mediaFilter +=  "%s files (*.%s)|*.%s;*.%s;|" % (media.upper(), \
                media.lower(), media.lower(), media.upper())
        mediaFilter += "All files (*) |*;|"
        fdlb = wx.FileDialog(self, "Select media files for the play-list...", \
            wildcard=mediaFilter, style=wx.FD_OPEN | \
            wx.FD_FILE_MUST_EXIST | wx.FD_MULTIPLE | wx.FD_PREVIEW)
        ret = fdlb.ShowModal()
        if ret == wx.ID_OK:
            paths = fdlb.GetPaths()
            for path in paths:
                self.__playListWindow.addSongsToPlayList( \
                    [MediaSong.MediaSong(path)])

    def __onAddSongToPlayList(self, event):
        """ Adds the selected song to the play-list

         The selected songs in the search window are placed into the play-list.
         wx.Event event: is the event object (Not Used)
        """
        self.__playListWindow.addSongsToPlayList( \
            self.__searchWindow.selectedSongs())

    def __onPlaySong(self, event):
        """ Plays a song from the play-list or pauses the player.

         A song is selected from the play-list, pauses the player if it is
           already playing a song, or un-pauses the player if it is already
           paused.
         wx.Event event: is the event object (Not Used)
        """
        if self.__playerFactory.player.isPlaying():
            self.__playerFactory.player.pauseSong()
            self.__playerWindow.togglePlay()
        elif self.__playerFactory.player.isPaused():
            self.__playerFactory.player.unpauseSong()
            self.__playerWindow.togglePlay()
        else:
            self.__handleStartSong()

    def __onRestartSong(self, event):
        """ Restarts the current playing song. """
        if self.__playerFactory.player.isPlaying() or \
                self.__playerFactory.player.isPaused():
            currentSong = self.__currentSong
            self.__playerFactory.player.stopSong()
            self.__handleEndSong()
            self.__handleStartSong(currentSong)

    def __handleStartSong(self, song=None):
        """ Handles the details of starting of a song.

         Selects a song from the play-list based on the current settings and
           hands off to the player the song, removing it from the play-list.
           The song is selected either by direct input from the user, by
           random, or by continuous play from the top of the play-list. Also,
           sets the player interface to reflect the current state.
         MediaSong song: is the specific song to play, else pulls from the
           play-list.
        """
        # Determine which song to play
        if song is None:
            if self.__playRandom:
                song = self.__playListWindow.playRandomSong( \
                    self.__removeFromPlayList)
            elif self.__playContinuous:
                song = self.__playListWindow.playFirstSong( \
                    self.__removeFromPlayList)
            else:
                song = self.__playListWindow.playSelectedSong( \
                    self.__removeFromPlayList)

        # Try to play the song
        if song.filePath is not None:
            if self.__playerFactory.player.playSong( \
                    song, self.__volumeLevel):
                self.__playerWindow.togglePlay()
                self.StatusBar.SetStatusText("Loaded: %s - %s" % (song.title, \
                    song.artist), self.__PLAYER_STATUS_FIELD)
                self.__currentSong = song
            else:
                wx.MessageBox("Could not start player or load media file!", \
                    "Player Error", wx.OK | wx.ICON_ERROR | wx.STAY_ON_TOP)
                if song.filePath and not self.__removeFromPlayList:
                    self.__playListWindow.addSongsToPlayList([song])
                self.__currentSong = None

    def __onStopSong(self, event):
        """ Stops the currently playing song.

         wx.Event event: is the event object (Not Used)
        """
        self.__playerFactory.player.stopSong()
        self.__handleEndSong()

    def __onChangeVolume(self, event):
        """ Changes the player volume levels.

         Updates the player volume levels from the player interface.
         wx.Event event: is the event object (Not Used)
        """
        newVolume = self.__playerWindow.volumeLevel()
        if self.__playerFactory.player.setVolumeLevel(newVolume):
            self.__volumeLevel = newVolume
        else:
            self.__playerWindow.setVolumeLevel(self.__volumeLevel)

    def __onChangeTrackPosition(self, event):
        """ Changes the player track position.

         Updates the player track position from the player interface.
         wx.Event event: is the event object (Not Used)
        """
        if self.__playerFactory.player.isPlaying():
            self.__playerFactory.player.setPlayPosition( \
                self.__playerWindow.trackPosition())
        elif not self.__playerFactory.player.isPaused():
            self.__playerWindow.setTrackPosition(0, 0)

    def __updateTrackPosition(self, event):
        """ Updates the player interface's track position display.

         Changes the player interface to reflect the current play position
           of the player.
         wx.Event event: is the event object, method only runs if the ID is
           self.__ID_TRACK_TIMER
        """
        if not event.GetId() == self.__ID_TRACK_TIMER:
            return
        else:
            if self.__playerFactory.player.isPlaying() or \
                    self.__playerFactory.player.isPaused():
                self.__playerWindow.setTrackPosition( \
                    self.__playerFactory.player.positionInSong(), \
                    self.__playerFactory.player.totalSongTime())
            elif self.__playerFactory.player.isSongEnded() and not \
                    self.__playerWindow.isDisplayingPlay():
                self.__onEndSong(-1) # TODO should be done as an event.

    def __handleEndSong(self):
        """ Handles the interface details of stopping a song.

         Stops the currently playing song, and updates the player interface.
        """
        self.__playerWindow.setTrackPosition(0, 0)
        if not self.__playerWindow.isDisplayingPlay():
            self.__playerWindow.togglePlay()
        self.StatusBar.SetStatusText("Player Ready", \
            self.__PLAYER_STATUS_FIELD)
        self.__currentSong = None

    def __onEndSong(self, event):
        """ Handles the end of a song event.

         Updates the player interface and will select another song to play if
           continuous play has been set.
         wx.Event event: is the event object (Not Used)
        """
        self.__handleEndSong()
        if self.__playContinuous:
            self.__handleStartSong()

    def __onExit(self, event):
        """ Handles the exiting of the programme.

         Handles the closing of the windows, updates the configuration, and
           ensures the player is not left in an unknown state.
         wx.Event event: is the event object (Not Used)
        """
        if self.__playerFactory.player.isRunning():
            self.__playerFactory.player.quitRunning()
        self.__writeSettings()
        if self.__database.isRunning():
            self.__database.quitRunning()
        self.Show(False)
        self.Destroy()

    def __onAbout(self, event):
        """ Displays version and license information for Kap

         wx.Event event: is the event object (Not Used)
        """
        abtnfAbout = wx.AboutDialogInfo()
        #abtnfAbout.AddArtist("N/A")
        abtnfAbout.SetCopyright("(C) 2009 John Schneiderman")
        abtnfAbout.SetDescription( \
            "A player with a search-able media library.")
        abtnfAbout.AddDeveloper("John Schneiderman <JohnMS@member.fsf.org>")
        #abtnfAbout.AddDocWriter("N/A")
        #abtnfAbout.SetIcon(icon)
        GPLv3_Notice = "Kap is free software; you can redistribute it and/or"
        GPLv3_Notice += "\nmodify it under the terms of the GNU General "
        GPLv3_Notice += "Public\nLicense as published by the Free Software "
        GPLv3_Notice += "Foundation; either\nversion 3 of the License, or "
        GPLv3_Notice += "(at your option) any later version.\n\nKap is "
        GPLv3_Notice += "distributed in the hope that it will be useful,\nbut "
        GPLv3_Notice += "WITHOUT ANY WARRANTY; without even the implied"
        GPLv3_Notice += " warranty of\nMERCHANTABILITY or FITNESS FOR A"
        GPLv3_Notice += " PARTICULAR PURPOSE.  See the\nGNU General Public"
        GPLv3_Notice += " License for more details.\n\nYou should have "
        GPLv3_Notice += "received a copy of the GNU General Public\nLicense"
        GPLv3_Notice += " along with this program; if not, write to the Free"
        GPLv3_Notice += " Software\nFoundation, Inc., 59 Temple Place, Suite"
        GPLv3_Notice += " 330, Boston, MA  02111-1307  USA"
        abtnfAbout.SetLicence(GPLv3_Notice)
        abtnfAbout.SetName("Karaoke Audio-video Player")
        #abtnfAbout.AddTranslator("N/A")
        abtnfAbout.SetVersion(_VERSION)
        abtnfAbout.SetWebSite("http://kap.nongnu.org")
        wx.AboutBox(abtnfAbout)

    def __onMenuPlayer(self, event):
        """ Updates the settings for playing from the play-list.

         wx.Event event: is the event object (Not Used)
        """
        self.__playContinuous = self.__mnPlayer.IsChecked(self.__ID_CONTINUOUS)
        self.__playRandom = self.__mnPlayer.IsChecked(self.__ID_RANDOM)

def main(args):
    """ Launcher for the Kap stand-alone player.

     Currently there are no command line options.
    """

    # Display license
    print "Kap version %s Copyright (C) 2009 John Schneiderman" % _VERSION
    print "Kap comes with ABSOLUTELY NO WARRANTY;"
    print "This is free software, and you are welcome to redistribute it"
    print "under certain conditions; see the COPYING file for details,"
    print "or the Free Software Foundation's GPL.\n"

    pySmplAppKap = wx.PySimpleApp()
    sfCnfgPrsrKap = ConfigParser.SafeConfigParser()
    sfCnfgPrsrKap.read('kap.cfg')
    kap = Kap(None, wx.ID_ANY, sfCnfgPrsrKap)
    kap.Show(True)
    pySmplAppKap.MainLoop()
    configfile = open('kap.cfg', 'wb')
    sfCnfgPrsrKap.write(configfile)

if __name__ == "__main__":
    main(sys.argv)
