#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import ConfigParser
import os
import sys
import fnmatch
import cPickle

from mediasong import MediaSong


class MediaLibrary():
    """ Database that contains all the found audio and video music files. """
    # TODO Use a list of the supported media file types, each marked as audio
    #   or video.

    """
     ATTRIBUTES
    """
    # The path to the music database
    __databasePath = "."
    # All the songs in the database
    __songs = []
    # All the counts for each song
    __songCounts = {}
    # All the full search paths in the database
    __searchPaths = []
    # Determines if the database needs to be saved
    __blIsDirty = False

    def searchPaths(self):
        """ Accessor to the paths to search for new media.

         return list[string]: The search paths for new media.
        """
        return self.__searchPaths

    def readSettings(self, configuration):
        """ Reads the media library settings.

         Reads in the database path, and the search paths, with each search
           path separated by a ';'
         SafeConfigParser configuration: holds the media library settings.
        """
        if configuration.has_section('MediaLibrary'):
            if configuration.has_option('MediaLibrary', "databasePath"):
                self.__databasePath = configuration.get("MediaLibrary", \
                    "databasePath")
            if configuration.has_option('MediaLibrary', "searchPaths"):
                self.__searchPaths = configuration.get('MediaLibrary', \
                    'searchPaths').split(";")
        try:
            file = open( \
                "%s/media_library.database" % self.__databasePath, "r")
        except IOError, message:
            print >> sys.stderr, "Song database could not be opened:", message
        else:
            self.__songs = cPickle.load(file)
            file.close()

        try:
            file = open( \
                "%s/media_library_count.database" % self.__databasePath, "r")
        except IOError, message:
            print >> sys.stderr, "Song count database could not be opened:", \
                message
        else:
            self.__songCounts = cPickle.load(file)
            file.close()

    def writeSettings(self, configuration):
        """ Writes out the media library settings.

         Sets the database path and the search path with each search path
           separated by a ';'
         SafeConfigParser configuration: is the object to save the settings in.
        """
        if self.__blIsDirty:
            print "MediaLibrary.writeSettings(configuration)", configuration
            if not configuration.has_section('MediaLibrary'):
                configuration.add_section('MediaLibrary')
            configuration.set('MediaLibrary', 'databasePath', \
                self.__databasePath)
            # Save the search paths by separating them by ;
            if self.__searchPaths is not None:
                paths = ''
                for path in self.__searchPaths:
                    if path is not None:
                        paths += path + ";"
                configuration.set('MediaLibrary', 'searchPaths', paths)

            try:
                file = open( \
                    "%s/media_library.database" % self.__databasePath, "w")
            except IOError, message:
                print >> sys.stderr, "Song database could not be opened:", \
                    message
            else:
                cPickle.dump(self.__songs, file)
                file.close()

            try:
                file = open( \
                    "%s/media_library_count.database" % self.__databasePath, \
                    "w")
            except IOError, message:
                print >> sys.stderr, "Count database could not be opened:", \
                    message
            else:
                cPickle.dump(self.__songCounts, file)
                file.close()

    def supportedMedia(self):
        """ All the supported media file extensions.

         return list[string]: the list of all media file extensions.
        """
        return ['ogg', 'flac', 'mp3', 'mp2', 'wma', 'wav', 'mov', 'wmv', \
            'avi', 'flv', 'mpeg', 'mpg', 'mpg']

    def buildLibrary(self, paths):
        """ Builds the database of all the audio and video files.

         Creates a new database with the files listed in the search path. If
           a file does not contain an extension of one of the supported media
           formats a message is given to indicate it not being included.
         list[strings] paths: is the paths to the directory to search for all
           audio and video music files.
         return int: the number of songs found
        """
        self.__searchPaths = paths
        del self.__songs[:]
        self.__songCounts.clear()
        found = 0
        for path in paths:
            uncdPath = path.encode('UTF-8', 'replace')
            for root, dirs, files in os.walk(uncdPath):
                for name in files:
                    if fnmatch.fnmatch(name.lower(), '*.mp3') or \
                            fnmatch.fnmatch(name.lower(), '*.ogg') or \
                            fnmatch.fnmatch(name.lower(), '*.flac') or \
                            fnmatch.fnmatch(name.lower(), '*.mp2') or \
                            fnmatch.fnmatch(name.lower(), '*.wma') or \
                            fnmatch.fnmatch(name.lower(), '*.wav'):
                        self.__addSong("%s/%s" % (root, name), False)
                        found += 1
                    elif fnmatch.fnmatch(name.lower(), '*.mov') or \
                            fnmatch.fnmatch(name.lower(), '*.mpg') or \
                            fnmatch.fnmatch(name.lower(), '*.avi') or \
                            fnmatch.fnmatch(name.lower(), '*.flv') or \
                            fnmatch.fnmatch(name.lower(), '*.mpeg') or \
                            fnmatch.fnmatch(name.lower(), '*.asf') or \
                            fnmatch.fnmatch(name.lower(), '*.wmv'):
                        self.__addSong("%s/%s" % (root, name), True)
                        found += 1
                    else:
                        print "Format not supported: %s/%s" % (root, name)

        return found

    def __addSong(self, filename, isVideo):
        """ Attempts to add a song to the media library

         Places a MediaSong object into the database, and marks the
           database as dirty. If a song cannot be added, a message stating
           what file could not be added is given.
         string filename: is the absolute path and name of the file to parse
           into a MediaSong object.
         boolean isVideo: indicates if the file is a video file. True if it
           is, false else-wise.
        """
        song = MediaSong(filename, isVideo)
        if song.artist is not None and song.title is not None:
            self.__songs.append(song)
            self.__songCounts[song.filePath] = 0
            self.__blIsDirty = True
        else:
            print "Could not determine artist or tile: %s" % song.filePath

    def incrementSongPlayed(self, song):
        """ Increments the song counter.

         Increments the song counter by one.
         MediaSong song: is the song that is to be played.
        """
        self.__songCounts[song.filePath] += 1

    def findSongs(self, searchTerms, searchAudio, searchVideo):
        """ Finds songs that meet a given criteria.

         Searches through the database and finds songs that meet all of the
           terms the user entered.
         string searchTerms: is the search items in part or in whole of the
           name of a song either by artist or by title. Each term is separated
           by white space. For inclusion in the results, a song must contain
           all the search terms, case insensitive, and there must be terms to
           compare with.
         boolean searchAudio: True if the song must be an audio media.
         boolean searchVideo: True if the song must be a video media.
         list[MediaSong] : all the songs found that match the search criteria
           as MediaSong items. If no song is found an empty list is given.
        """
        print "MediaLibrary.findSongs(searchTerms, searchAudio, searchVideo)", searchTerms, searchAudio, searchVideo
        songsFound = []
        if searchTerms:
            terms = searchTerms.split(" ")
            if terms:
                for song in self.__songs:
                    missingTerm = False

                    if song.blIsVideo and not searchVideo:
                        continue
                    elif not song.blIsVideo and not searchAudio:
                        continue
                    else:
                        for term in terms:
                            term = term.encode('UTF-8', 'replace').lower()
                            title = song.title.lower()
                            artist = song.artist.lower()
                            if (term not in title) and (term not in artist):
                                missingTerm = True
                                break

                        if not missingTerm:
                            songsFound.append(song)
        else:
            for song in self.__songs:
                if song.blIsVideo and not searchVideo:
                    continue
                elif not song.blIsVideo and not searchAudio:
                    continue
                else:
                    songsFound.append(song)
        return songsFound
