#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
from threading import Thread, Lock, Condition
import Queue
import playercontroller as Player
import playersignal as Signaller
import os


class PlayerThread(Thread):
    """ Run a media player in a non-blocking manner. """

    """
     ATTRIBUTES
    """
    # Player command queue
    __commandQueue = Queue.Queue()
    __signal = None
    __player = None

    """
     PLAYER STATES
    """
    # Indicates if the player is in a starting state.
    __blIsStarting = False
    # Indicates if the player is in a running state.
    __blIsRunning = False
    # Indicates if the player is in a playing state.
    __blIsPlaying = False
    # Indicates if the player is in a paused state.
    __blIsPaused = False
    # Indicates if a song has ended.
    __blIsSongEnded = False

    """
     THREAD LOCKS
    """
    # Lock for accessing the command queue.
    __lckCommandQueue = None
    # Lock for running a command from the command queue.
    __lckRunningCommand = None
    # Condition for running a command from the command queue.
    __cndtnRunCommand = None

    """
     SIGNAL COMMANDS
    """
    # Indicates a command has not been issued.
    __commandEmpty = hex(0x0)
    # Indicates the player should quit running.
    __commandQuitRunning = hex(0x1)
    # Indicates the player should play the set song.
    __commandPlay = hex(0x2)
    # Indicates the player should pause the playing song.
    __commandPause = hex(0x3)
    # Indicates the player should un-pause the playing song.
    __commandUnpause = hex(0x4)
    # Indicates the player should stop the playing song.
    __commandStop = hex(0x5)
    # Indicates the player should change the volume level.
    __commandVolume = hex(0x6)
    # Indicates the player should set the total time.
    __commandTotalTime = hex(0x7)
    # Indicates the player should set the playing position.
    __commandPositionTime = hex(0x8)
    # Indicates the player should change the playing position.
    __commandSetPositionTime = hex(0x9)
    # Indicates the player has reached the end of a song.
    __commandSongEnd = hex(0xA)
    # Indicates the player should display the song at full screen.
    __commandFullScreen = hex(0xB)

    def __init__(self, playerSignaller):
        """ Create and start the player thread.

         Creates and starts the player thread and initialises all thread locks.
         PlayerSignal playerSignaller: is the signal handeler for the media
           player.
        """
        Thread.__init__(self, name=playerSignaller.name())
        print "PlayerThread.__init__(self, playerSignaller)", self, \
            playerSignaller
        self.__signal = playerSignaller
        self.__player = Player.PlayerController()

        # Create the locks and conditions
        self.__lckRunningCommand = Lock()
        self.__lckCommandQueue = Lock()
        self.__cndtnRunCommand = Condition(self.__lckRunningCommand)
        self.start()

    def run(self):
        """ Process command queue and issue player signals.

         Processes __commandQueue and issue any player signals to the player
           until __commandQuitRunning is placed in the queue and processed.
        """
        self.__blIsRunning = True
        print "Player Thread Started"
        while self.__blIsRunning:
            self.__cndtnRunCommand.acquire()
            command = self.__commandEmpty

            while self.__commandQueue.empty():
                # Determine if the song has ended
                if self.__blIsPlaying and (self.__signal.totalTime > 0.0):
                    if self.__player.isPlayerReady() or \
                            (self.__signal.positionTime == \
                            self.__signal.totalTime):
                        self.__pushCommand(self.__commandSongEnd)
                self.__cndtnRunCommand.wait()

            self.__lckCommandQueue.acquire()
            command = self.__commandQueue.get(True)
            self.__lckCommandQueue.release()
            print " * Player Thread Processing Command:", command

            # Signal to quit the player
            if command == self.__commandQuitRunning:
                print " * Player Thread Running Signal Quit"
                if self.__blIsPlaying or self.__blIsPaused:
                    self.__blIsPlaying = False
                    self.__blIsPaused = False
                    self.__blIsSongEnded = False
                    self.__signal.quit_(self.__player)
                self.__lckCommandQueue.acquire()
                while not self.__commandQueue.empty():
                    self.__commandQueue.get(True)
                self.__lckCommandQueue.release()
                self.__blIsRunning = False
                print " * Player Thread End Of Signal Quit"

            # Signal to play the set song
            if command == self.__commandPlay:
                if not self.__blIsPlaying and not self.__blIsPaused:
                    self.__blIsStarting = True
                    print " * Player Thread Running Signal Play"
                    self.__signal.play(self.__player)
                    self.__blIsPlaying = True
                    self.__blIsSongEnded = False
                    self.__blIsStarting = False
                print " * Player Thread End Of Signal Play"

            # Signal to display song at full screen
            if command == self.__commandFullScreen:
                print " * Player Thread Running Signal FullScreen"
                self.__signal.fullScreen(self.__player)
                print " * Player Thread End Of Signal FullScreen"

            # Signal to stop the song
            if command == self.__commandStop:
                print " * Player Thread Running Signal Stop"
                self.__blIsPlaying = False
                self.__blIsPaused = False
                self.__signal.stop(self.__player)
                print " * Player Thread End Of Signal Stop"

            # Signal end of the song
            if command == self.__commandSongEnd:
                print " * Player Thread Running Signal Song End:"
                self.__blIsPlaying = False
                self.__blIsPaused = False
                self.__blIsSongEnded = True
                self.__signal.volumeLevel = 0
                self.__signal.totalTime = 0.0
                self.__signal.positionTime = 0.0
                self.__signal.quit_(self.__player)
                print " * Player Thread End Of Signal Song End"

            # Signal to pause the song
            if command == self.__commandPause:
                print " * Player Thread Running Signal Pause"
                self.__signal.pause(self.__player)
                print " * Player Thread End Of Signal Pause"

            # Signal to unpause the song
            if command == self.__commandUnpause:
                print " * Player Thread Running Signal Unpause"
                self.__signal.unpause(self.__player)
                print " * Player Thread End Of Signal Unpause"

            # Signal to change the volume of the player
            if command == self.__commandVolume: # set volume
                print " * Player Thread Running Signal Volume"
                self.__signal.setVolume(self.__player)
                print " * Player Thread End Of Signal Volume"

            # Signal to get the total time of the song.
            if command == self.__commandTotalTime: # get total time
                print " * Player Thread Running Signal TotalTime"
                self.__signal.playerTotalTime(self.__player)
                print " * Player Thread End Of Signal TotalTime"

            # Signal to get the current position of the song.
            if command == self.__commandPositionTime:
                # Check is placed here to ensure we can correctly determine
                #   the end of a song.
                if not self.__blIsPaused:
                    print " * Player Thread Running Signal PositionTime"
                    self.__signal.playerPositionTime(self.__player)
                    print " * Player Thread End Of Signal PositionTime"

            # Signal to set the current position of the song.
            if command == self.__commandSetPositionTime:
                print " * Player Thread Running Signal SetPostionTime"
                self.__signal.setPlayerPositionTime(self.__player)
                print " * Player Thread End Of Signal SetPostionTime"
            print "Player Thread Finished Command:", command
            self.__cndtnRunCommand.release()
        print "Player Thread Ended"

    def __pushCommand(self, command):
        """ Adds a signal command to the command queue

         SignalCommand command: is the command to issue as a signal.
        """
        self.__lckCommandQueue.acquire()
        if self.__blIsRunning:
            print "Player Thread Adding Command:", command
            self.__commandQueue.put(command, True)
        self.__lckCommandQueue.release()

    def isRunning(self):
        """ Indicates if the player is in a running state. """
        return self.__blIsRunning

    def isStarting(self):
        """ Indicates if the player is in a starting state. """
        return self.__blIsStarting

    def isSongEnded(self):
        """ Indicates if a song has ended.

         return boolean: True if the player has finished a playing song,
           False else-wise
        """
        return self.__blIsSongEnded

    def isPlaying(self):
        """ Indicates if the player is in a playing state. """
        return self.__blIsPlaying

    def isPaused(self):
        """ Indicates if the player is in a paused state. """
        return self.__blIsPaused

    def quitRunning(self):
        """ Issues the command to quit running. """
        print "PlayerThread.quitRunning(self)", self
        self.__pushCommand(self.__commandQuitRunning)
        self.__cndtnRunCommand.acquire()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def fullScreen(self):
        """ Issues the command to display the current song full screen. """
        print "PlayerThread.fullScreen(self)", self
        self.__pushCommand(self.__commandFullScreen)
        self.__cndtnRunCommand.acquire()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def play(self, filePath, volume):
        """ Issues the command to play a song.

         string filePath: is the path to the file to play.
         int volume: is the percentage volume level to play the song at.
         return boolean: True if commands are placed in the queue, False
           if a song is already loaded or if the file cannot be found.
        """
        print "PlayerThread.play(self, filePath, volume)", self, filePath, \
            volume
        if not filePath:
            print "PlayerThread.play(): Empty filePath"
            return False
        elif not os.access(filePath, os.F_OK):
            print "PlayerThread.play(): File not found: %s " % filePath
            return False
        elif not os.access(filePath, os.R_OK):
            print "PlayerThread.play(): missing read access: %s " % filePath
            return False
        elif not self.__blIsPlaying and not self.__blIsPaused:
            self.__signal.songFilePath = " \"" + filePath + "\""
            self.__pushCommand(self.__commandPlay)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
            return self.setVolumeLevel(volume, noCheckPlaying=True)
        else:
            print "PlayerThread.play(): Already playing a song."
            return False

    def stop(self):
        """ Issues the command to stop. """
        print "PlayerThread.stop(self)", self
        if not self.__blIsSongEnded and (self.__blIsPlaying or \
                self.__blIsPaused):
            self.__pushCommand(self.__commandStop)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()

    def pause(self):
        """ Issues the command to pause """
        print "PlayerThread.pause(self)", self
        if self.__blIsPlaying:
            self.__pushCommand(self.__commandPause)
            self.__blIsPlaying = False
            self.__blIsPaused = True
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()

    def unpause(self):
        """ Issues the command to un-pause """
        print "PlayerThread.unpause(self)", self
        if self.__blIsPaused:
            self.__pushCommand(self.__commandUnpause)
            self.__blIsPlaying = True
            self.__blIsPaused = False
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()

    def setVolumeLevel(self, volume, noCheckPlaying=False):
        """ Issues the command to set the volume level.

         int volume: is the volume level percentage.
         boolean noCheckPlaying: by passes the requirement to only change the
           volume while the player is playing. Use with caution.
         return boolean: True if the volume command is placed in the queue,
           False if we are not in a playing state.
        """
        print "PlayerThread.setVolumeLevel(self, volume, noCheckPlaying)", \
            self, volume, noCheckPlaying
        if self.__blIsPlaying or noCheckPlaying:
            self.__signal.volumeLevel = volume
            self.__pushCommand(self.__commandVolume)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
            return True
        else:
            return False

    def volumeLevel(self):
        """ Gives the current volume level percentage. """
        return self.__signal.volumeLevel

    def positionInSong(self):
        """ Give the current position in the playing song.

         Places the command to update the position of the playing song, only
           when in a playing state; condition verified in thread.
         return float: the current seconds into a song.
        """
        self.__pushCommand(self.__commandPositionTime)
        self.__cndtnRunCommand.acquire()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()
        return self.__signal.positionTime

    def setPlayPosition(self, seconds):
        """ Issues the command to set the current playing position.

         float seconds: is the number of seconds to play from.
         return boolean: True if we are in a playing state, False else-wise.
        """
        print "PlayerThread.setPlayPosition(self, seconds)", self, seconds
        if self.__blIsPlaying:
            self.__signal.positionTime = seconds
            self.__pushCommand(self.__commandSetPositionTime)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
            return True
        else:
            return False

    def totalSongTime(self):
        """ Give the total time in the playing song.

         Places the command to update the total time of the playing song, only
           when in a playing state.
         return float: the total seconds of the loaded song.
        """
        if self.__blIsPlaying:
            self.__pushCommand(self.__commandTotalTime)
            self.__cndtnRunCommand.acquire()
            self.__cndtnRunCommand.notify()
            self.__cndtnRunCommand.release()
        return self.__signal.totalTime
