#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import ConfigParser
import sys
import cPickle

from mediasong import MediaSong

import medialibrarybase as Base


class MediaLibraryFile(Base.MediaLibraryBase):
    """ Database that contains all the found audio and video music files. """

    """
     ATTRIBUTES
    """
    # All the songs in the database
    __songs = []
    # All the counts for each song
    __songCounts = {}
    # Determines if the database needs to be saved
    __blIsDirty = False

    def readSettings(self, configuration):
        """ Reads the media library settings.

         Reads in the database path, and the search paths, with each search
           path separated by a ';'
         SafeConfigParser configuration: holds the media library settings.
        """
        Base.MediaLibraryBase.readSettings(self, configuration)

        try:
            file = open("%s/%s" % (Base.MediaLibraryBase.databasePath, \
                    Base.MediaLibraryBase.DATABASE_FILE_NAME), "r")
        except IOError, message:
            print >> sys.stderr, "Song database could not be opened:", message
        else:
            try:
                self.__songs = cPickle.load(file)
            except EOFError, message:
                self.__songs = []
                print >> sys.stderr, "Song database could not be loaded:", \
                        message
            finally:
                file.close()

        try:
            file = open( \
                "%s/media_library_count.database" % self.databasePath, "r")
        except IOError, message:
            print >> sys.stderr, "Song count database could not be opened:", \
                message
        else:
            try:
                self.__songCounts = cPickle.load(file)
            except EOFError, message:
                self.__songCounts = []
                print >> sys.stderr, \
                        "Song count database could not be loaded:", message
            finally:
                file.close()

    def writeSettings(self, configuration):
        """ Writes out the media library settings.

         Sets the database path and the search path with each search path
           separated by a ';'
         SafeConfigParser configuration: is the object to save the settings in.
        """
        if self.__blIsDirty:
            Base.MediaLibraryBase.writeSettings(self, configuration)

            try:
                file = open("%s/%s" % (Base.MediaLibraryBase.databasePath, \
                        Base.MediaLibraryBase.DATABASE_FILE_NAME), "w")
            except IOError, message:
                print >> sys.stderr, "Song database could not be opened:", \
                    message
            else:
                cPickle.dump(self.__songs, file)
                file.close()

            try:
                file = open( \
                    "%s/media_library_count.database" % self.databasePath, \
                    "w")
            except IOError, message:
                print >> sys.stderr, "Count database could not be opened:", \
                    message
            else:
                cPickle.dump(self.__songCounts, file)
                file.close()

    def buildLibrary(self, paths):
        """ Builds the database of all the audio and video files.

         Creates a new database with the files listed in the search path. If
           a file does not contain an extension of one of the supported media
           formats a message is given to indicate it not being included.
         list[strings] paths: is the paths to the directory to search for all
           audio and video music files.
         return int: the number of songs found
        """
        self.pathSearchList = paths[:]
        foundSongs = Base.MediaLibraryBase.locateMediaSongs(self)
        del self.__songs[:]
        self.__songCounts.clear()

        found = 0
        for song in foundSongs:
            path, isVideo = song
            self.__addSong(path, isVideo)
            found += 1

        return found

    def __addSong(self, filename, isVideo):
        """ Attempts to add a song to the media library

         Places a MediaSong object into the database, and marks the
           database as dirty. If a song cannot be added, a message stating
           what file could not be added is given.
         string filename: is the absolute path and name of the file to parse
           into a MediaSong object.
         boolean isVideo: indicates if the file is a video file. True if it
           is, false else-wise.
        """
        song = MediaSong(filename, isVideo)
        if song.artist is not None and song.title is not None:
            self.__songs.append(song)
            self.__songCounts[song.filePath] = 0
            self.__blIsDirty = True
        else:
            print "Could not determine artist or tile: %s" % song.filePath

    def incrementSongPlayed(self, song):
        """ Increments the song counter.

         Increments the song counter by one.
         MediaSong song: is the song that is to be played.
        """
        self.__songCounts[song.filePath] += 1
        self.__blIsDirty = True

    def findSongs(self, searchTerms, searchAudio, searchVideo, favourite=None):
        """ Finds songs that meet a given criteria.

         Searches through the database and finds songs that meet all of the
           terms the user entered.
         string searchTerms: is the search items in part or in whole of the
           name of a song either by artist or by title. Each term is separated
           by white space. For inclusion in the results, a song must contain
           all the search terms, case insensitive, and there must be terms to
           compare with.
         boolean searchAudio: True if the song must be an audio media.
         boolean searchVideo: True if the song must be a video media.
         boolean favourite: True if the song must be a favourite, False if it
           must not be a favourite, None if the condition does not matter.
         list[MediaSong]: all the songs found that match the search criteria
           as MediaSong items. If no song is found an empty list is given.
        """
        songsFound = []
        if searchTerms:
            terms = searchTerms.split(" ")
            if terms:
                for song in self.__songs:
                    missingTerm = False

                    if song.blIsVideo and not searchVideo:
                        continue
                    elif not song.blIsVideo and not searchAudio:
                        continue
                    elif favourite is not None and \
                            (song.blIsFavourite != favourite):
                        continue
                    else:
                        for term in terms:
                            term = term.encode('UTF-8', 'replace').lower()
                            title = song.title.lower()
                            artist = song.artist.lower()
                            if (term not in title) and (term not in artist):
                                missingTerm = True
                                break

                        if not missingTerm:
                            songsFound.append(song)
        else:
            for song in self.__songs:
                if song.blIsVideo and not searchVideo:
                    continue
                elif not song.blIsVideo and not searchAudio:
                    continue
                elif favourite is not None and (song.blIsFavourite != \
                        favourite):
                    continue
                else:
                    songsFound.append(song)
        return songsFound

    def setFavourite(self, song, favourite):
        """ Sets a specific songs favourite status.

         MediaSong song: is the song whose favourite status we are setting.
         boolan favourite: is the favourite status of the song
         boolean: True if the song was found and set, False else-wise.
        """
        for testSong in self.__songs:
            if testSong == song:
                testSong.blIsFavourite = favourite
                self.__blIsDirty = True
