#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import metakit

import medialibrarybase as Base
import mediasong as SongStruct


class MediaLibraryMetakit(Base.MediaLibraryBase):

    """
     ATTRIBUTES
    """
    # Database storage connection
    __database = None
    # Database storage view
    __songs = None

    def readSettings(self, configuration):
        """ Reads the media library settings.

         Reads in the database path, and the search paths, with each search
           path separated by a ';'
         SafeConfigParser configuration: holds the media library settings.
        """
        Base.MediaLibraryBase.readSettings(self, configuration)

        try:
            self.__database = metakit.storage("%s/%s" % ( \
                    Base.MediaLibraryBase.databasePath, \
                    Base.MediaLibraryBase.DATABASE_FILE_NAME), 2)
            self.__songs = self.__database.view("songs")
            if len(self.__songs) == 0:
                self.__songs = self.__database.getas("songs[filePath:S," + \
                        "artist:S,title:S,playCount:L,isVideo:I,isFavourite:I]")
        except IOError, message:
            print >> sys.stderr, "Song database could not be opened:", message

    def writeSettings(self, configuration):
        """ Writes out the media library settings.

         Sets the database path and the search path with each search path
           separated by a ';'
         SafeConfigParser configuration: is the object to save the settings in.
        """
        Base.MediaLibraryBase.writeSettings(self, configuration)

        self.__database.commit()

    def buildLibrary(self, paths):
        """ Builds the database of all the audio and video files.

         Creates a new database with the files listed in the search path. If
           a file does not contain an extension of one of the supported media
           formats a message is given to indicate it not being included.
         list[strings] paths: is the paths to the directory to search for all
           audio and video music files.
         return int: the number of songs found
        """
        self.__songs.remove(self.__songs[:])
        found = 0

        self.pathSearchList = paths[:]
        foundSongs = Base.MediaLibraryBase.locateMediaSongs(self)

        for song in foundSongs:
            path, isVideo = song
            mediaSong = SongStruct.MediaSong(path, isVideo)
            self.__songs.append(filePath=mediaSong.filePath, \
                    artist=mediaSong.artist, \
                    title=mediaSong.title, \
                    playCount=0, \
                    isVideo=mediaSong.blIsVideo, \
                    isFavourite=mediaSong.blIsFavourite)
            found += 1

        return found

    def incrementSongPlayed(self, song):
        """ Increments the song counter.

         Increments the song counter by one.
         MediaSong song: is the song that is to be played.
        """
        index = self.__songs.find(filePath = song.filePath)
        self.__songs[index].playCount += 1

    def findSongs(self, searchTerms, searchAudio, searchVideo, favourite=None):
        """ Finds songs that meet a given criteria.

         Searches through the database and finds songs that meet all of the
           terms the user entered.
         string searchTerms: is the search items in part or in whole of the
           name of a song either by artist or by title. Each term is separated
           by white space. For inclusion in the results, a song must contain
           all the search terms, case insensitive, and there must be terms to
           compare with.
         boolean searchAudio: True if the song must be an audio media.
         boolean searchVideo: True if the song must be a video media.
         boolean favourite: True if the song must be a favourite, False if it
           must not be a favourite, None if the condition does not matter.
         list[MediaSong]: all the songs found that match the search criteria
           as MediaSong items. If no song is found an empty list is given.
        """
        songs = []

        if searchTerms is None or (len(searchTerms) > 0):
            if favourite is None:
                if searchAudio:
                    searchQuery = self.__songs.select(isVideo=0)
                    for song in searchQuery:
                        songs.append(SongStruct.MediaSong(song.filePath, \
                                song.isVideo))
                if searchVideo:
                    searchQuery = self.__songs.select(isVideo=1)
                    for song in searchQuery:
                        songs.append(SongStruct.MediaSong(song.filePath, \
                                song.isVideo))
            elif favourite:
                if searchAudio:
                    searchQuery = self.__songs.select(isFavourite=1,isVideo=0)
                    for song in searchQuery:
                        songs.append(SongStruct.MediaSong(song.filePath, \
                                song.isVideo))
                if searchVideo:
                    searchQuery = self.__songs.select(isFavourite=1,isVideo=1)
                    for song in searchQuery:
                        songs.append(SongStruct.MediaSong(song.filePath, \
                                song.isVideo))
            else:
                if searchAudio:
                    searchQuery = self.__songs.select(isFavourite=0,isVideo=0)
                    for song in searchQuery:
                        songs.append(SongStruct.MediaSong(song.filePath, \
                                song.isVideo))
                if searchVideo:
                    searchQuery = self.__songs.select(isFavourite=0,isVideo=1)
                    for song in searchQuery:
                        songs.append(SongStruct.MediaSong(song.filePath, \
                                song.isVideo))

            if searchTerms is not None:
                terms = searchTerms.split(" ")
                songList = songs[:]
                for song in songList:
                    missingTerm = False
                    for term in terms:
                        term = term.encode('UTF-8', 'replace').lower()
                        title = song.title.lower()
                        artist = song.artist.lower()
                        if (term not in title) and (term not in artist):
                            missingTerm = True
                            break
                    if missingTerm:
                        songs.remove(song)
        else:
            searchQuery1 = self.__songs.select(isFavourite=0)
            searchQuery2 = self.__songs.select(isFavourite=1)
            for song in searchQuery1:
                if (searchAudio and not song.isVideo) or \
                        (searchVideo and song.isVideo):
                    songs.append(SongStruct.MediaSong(song.filePath, \
                            song.isVideo))
            for song in searchQuery2:
                if (searchAudio and not song.isVideo) or \
                        (searchVideo and song.isVideo):
                    songs.append(SongStruct.MediaSong(song.filePath, \
                            song.isVideo))

        return songs

    def setFavourite(self, song, favourite):
        """ Sets a specific songs favourite status.

         MediaSong song: is the song whose favourite status we are setting.
         boolan favourite: is the favourite status of the song
         boolean: True if the song was found and set, False else-wise.
        """
        index = self.__songs.find(filePath = song.filePath)

        if favourite:
            self.__songs[index].isFavourite = 1
        else:
            self.__songs[index].isFavourite = 0
