#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
from mediasong import MediaSong

import wx
import mediaparser as parser


class MediaSongWindow(wx.Panel):
    """ Interface to display a MedaSong. """

    """
     ATTRIBUTES
    """
    # A list of the MediaSong objects displayed.
    __mediaSongs = []
    # Is the width of the media column.
    __MEDIA_WIDTH = 50
    # Approximate width of the vertical scroll bar
    __VERTICAL_SCROLL_BAR = 25
    # Maximum number of songs to display at once
    __MAXIMUM_DISPLAYED_SONGS = 50
    # Current index position displayed
    __displayedStartIndex = None

    """
     EVENT ID
    """
    # ID of the object displaying the media songs
    __ID_DISPLAYED_SONGS = wx.NewId()
    # Column ID for the media type
    __ID_MEDIA_COLUMN = 0
    # Column ID for the song title
    __ID_TITLE_COLUMN = 1
    # Column ID for the artist title
    __ID_ARTIST_COLUMN = 2
    # Previous displayed page
    __ID_PREVIOUS = wx.NewId()
    # Next displayed page
    __ID_NEXT = wx.NewId()
    # Page navigation indicator
    __ID_PAGES = wx.NewId()
    # Used to signal to display the media songs file information
    __ID_FILE_INFORMATION = wx.NewId()

    def __init__(self, parent, iconPath, displayId=None):
        """ Default constructor

         wx.Window parent: is the parent window.
         string iconPath: is the base path to the icon directory
         integer displayID: is the ID for the object displaying the media
           songs. If it is not set, one is automatically generated.
        """
        wx.Panel.__init__(self, parent, wx.NewId())
        if displayId is not None:
            self.__ID_DISPLAYED_SONGS = displayId
        self.__createMenus()
        self.__createControls(iconPath)
        self.__bindEvents()
        self.__doLayout()

    def __createMenus(self):
        """ Create the file menu options.

         Creates the File, Player, Settings, and Help menus.
        """
        # Create right click pop-up for search/favourites window
        self.mnDisplayWindowPopup = wx.Menu()
        self.__mnItmFileInformation = wx.MenuItem(self.mnDisplayWindowPopup, \
            self.__ID_FILE_INFORMATION, 'View File Information')
        self.mnDisplayWindowPopup.AppendItem(self.__mnItmFileInformation)

    def __createControls(self, iconPath):
        """ Create the controls for the window.

        """
        # Create Media Song Display
        self. __lstCtlMediaSongs = wx.ListCtrl(self, \
            self.__ID_DISPLAYED_SONGS, style=wx.LC_REPORT | wx.SUNKEN_BORDER | \
             wx.LC_HRULES | wx.LC_VRULES)
        self. __lstCtlMediaSongs.InsertColumn(self.__ID_MEDIA_COLUMN, \
            "Media", self.__MEDIA_WIDTH)
        self. __lstCtlMediaSongs.InsertColumn(self.__ID_TITLE_COLUMN, \
            "Title")
        self. __lstCtlMediaSongs.InsertColumn(self.__ID_ARTIST_COLUMN, \
            "Artist")

        # Large list navigation
        self.__btmpBttnPrevious = wx.BitmapButton(self, self.__ID_PREVIOUS, \
            wx.Bitmap(iconPath + '/prev.png'))
        self.__btmpBttnPrevious.SetToolTip(wx.ToolTip( \
            "Display Previous Page"))
        self.__btmpBttnPrevious.Enable(False)
        self.__spnCtrlPage = wx.SpinCtrl(self, self.__ID_PAGES, \
            style=wx.TE_READONLY)
        self.__spnCtrlPage.SetRange(0, 0)
        self.__btmpBttnNext = wx.BitmapButton(self, self.__ID_NEXT, \
            wx.Bitmap(iconPath + '/next.png'))
        self.__btmpBttnNext.SetToolTip(wx.ToolTip( \
            "Display Next Page"))
        self.__btmpBttnNext.Enable(False)

    def __bindEvents(self):
        """ Connects all the needed events.

         Connects each of the control objects created to their corresponding
           methods.
        """
        wx.EVT_LIST_INSERT_ITEM(self, self.__ID_DISPLAYED_SONGS, \
            self.__onListInsertItem)

        # Page Navigation Events
        wx.EVT_BUTTON(self, self.__ID_PREVIOUS, self.__onPreviousPage)
        wx.EVT_BUTTON(self, self.__ID_NEXT, self.__onNextPage)
        wx.EVT_SPINCTRL(self, self.__ID_PAGES, self.__onPageChanged)

        wx.EVT_LIST_ITEM_RIGHT_CLICK(self, self.__ID_DISPLAYED_SONGS, \
            self.__onRightClick)
        wx.EVT_MENU(self, self.__ID_FILE_INFORMATION, \
            self.__onDisplaySongInformation)

    def __doLayout(self):
        """ Creates a visually pleasing look for the controls.

         All the control objects are placed on the window.
        """

        # Page Navigation Layout
        self.__bxSzrSearchNavigation = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrSearchNavigation.AddSpacer(5)
        self.__bxSzrSearchNavigation.Add(self.__spnCtrlPage, 0, wx.ALL, 3)
        self.__bxSzrSearchNavigation.AddSpacer(5)
        self.__bxSzrSearchNavigation.Add(self.__btmpBttnPrevious, 1, \
            wx.EXPAND | wx.ALL | wx.FIXED_MINSIZE, 3)
        self.__bxSzrSearchNavigation.AddSpacer(5)
        self.__bxSzrSearchNavigation.Add(self.__btmpBttnNext, 1, wx.EXPAND | \
            wx.ALL | wx.FIXED_MINSIZE, 3)

        # Media Song Display Layout
        self.__bxSzrMediaSongs = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrMediaSongs.Add(self.__lstCtlMediaSongs, 1, \
            wx.EXPAND | wx.ALL, 3)

        # Create Interface
        self.__bxSzrInterface = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrInterface.Add(self.__bxSzrMediaSongs, 1, wx.EXPAND | \
            wx.ALL)
        self.__bxSzrInterface.Add(self.__bxSzrSearchNavigation, 0)
        self.SetSizer(self.__bxSzrInterface)

    def displayedSongs(self):
        """ Accessor to the currently selected songs

         return list[MediaSong]: each of the songs selected by the user.
        """
        selection = []
        current = -1 # Assume invalid selection
        while True:
            next = self.__getNextDisplayedSong(current)
            if next == -1:
                return selection

            selection.append( \
                    self.__mediaSongs[next + self.__displayedStartIndex])
            current = next

    def __onRightClick(self, event):
        """ Handles view right click """
        self.PopupMenu(self.mnDisplayWindowPopup, event.GetPosition())

    def __onDisplaySongInformation(self, event):
        """ Handles the display of file informaiton """
        songs = self.displayedSongs()
        for song in songs:
            songInfo = parser.MediaParser(song)
            if songInfo.hasInformation():
                knownInformation = songInfo.information()
                info = ""
                for key in knownInformation.keys():
                    info += "  %s: %s\n" % (key, knownInformation[key])
                dlg = wx.MessageDialog(self, info,"Media Informaiton", \
                    wx.ICON_INFORMATION | wx.OK)
            else:
                dlg = wx.MessageDialog(self, "Unable to locate song %s." % \
                    song, "Media Informaiton Error", \
                    wx.ICON_INFORMATION | wx.OK)
            dlg.ShowModal()

    def __getNextDisplayedSong(self, current=-1):
        """ Get the next selected song.

         int current: is the current index.
         return int: the index of the next selected item, or -1 when no more
           selections are left.
        """
        return self. __lstCtlMediaSongs.GetNextItem(current, \
            wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)

    def clearDisplayedSongs(self):
        """ Clears the display window """
        del self.__mediaSongs[:]
        self. __lstCtlMediaSongs.DeleteAllItems()
        self.__displayedStartIndex = None

    def addDisplayedSong(self, song):
        """ Adds a media song to display.

         MediaSong song: is the song to add.
        """
        self.__mediaSongs.append(song)
        index = len(self.__mediaSongs) - 1
        if self.__displayedStartIndex is None:
            self.__addSongItem(index, song)
            self.__displayedStartIndex = 0
        elif len(self.__mediaSongs) <= self.__MAXIMUM_DISPLAYED_SONGS:
            self.__addSongItem(index, song)
        self.__updateNavigationButtons()

    def __addSongItem(self, index, song):
        """ Adds the song to the display by it's index. """

        # Title Information
        item = wx.ListItem()
        item.SetId(index)
        item.SetColumn(self.__ID_TITLE_COLUMN)
        item.SetText(song.title)
        self. __lstCtlMediaSongs.InsertItem(item)
        # Artist Information
        item = wx.ListItem()
        item.SetId(index)
        item.SetColumn(self.__ID_ARTIST_COLUMN)
        item.SetText(song.artist)
        self. __lstCtlMediaSongs.SetItem(item)
        # Video Information
        item = wx.ListItem()
        item.SetId(index)
        item.SetColumn(self.__ID_MEDIA_COLUMN)
        if song.blIsVideo:
            item.SetText("Video")
        else:
            item.SetText("Audio")
        self. __lstCtlMediaSongs.SetItem(item)

    def __onListInsertItem(self, event):
        """ Handles the column width display as each item is inserted.

         wx.Event event: is the event object (Not Used)
        """
        self.__adjustDisplayColumns()

    def __adjustDisplayColumns(self):
        """ Adjusts the width of the result columns. """
        width = self.__bxSzrInterface.GetSize().GetWidth() - \
            self.__MEDIA_WIDTH - self.__VERTICAL_SCROLL_BAR
        self. __lstCtlMediaSongs.SetColumnWidth(self.__ID_TITLE_COLUMN, \
            width / 2)
        self. __lstCtlMediaSongs.SetColumnWidth(self.__ID_ARTIST_COLUMN, \
            width / 2)
        self. __lstCtlMediaSongs.SetColumnWidth(self.__ID_MEDIA_COLUMN, \
            self.__MEDIA_WIDTH)

    def __onPreviousPage(self, event):
        """ Adjusts the media song display to previous maximum songs. """
        self.__displayedStartIndex -= self.__MAXIMUM_DISPLAYED_SONGS
        self.__displaySongsFromIndex(self.__displayedStartIndex)
        self.__updateNavigationButtons()

    def __onNextPage(self, event):
        """ Adjusts the media display to the maximum songs. """
        self.__displayedStartIndex += self.__MAXIMUM_DISPLAYED_SONGS
        self.__displaySongsFromIndex(self.__displayedStartIndex)
        self.__updateNavigationButtons()

    def __displaySongsFromIndex(self, startIndex):
        """ Updates the display of media songs from a starting index. """
        self. __lstCtlMediaSongs.DeleteAllItems()
        resultIndex = 0

        if (self.__displayedStartIndex + \
                self.__MAXIMUM_DISPLAYED_SONGS) < len(self.__mediaSongs):
            endIndex = (self.__MAXIMUM_DISPLAYED_SONGS + startIndex)
        else:
            endIndex = len(self.__mediaSongs)

        for index in range(startIndex, endIndex):
            self.__addSongItem(resultIndex, self.__mediaSongs[index])
            index += 1
            resultIndex += 1

    def __updateNavigationButtons(self):
        """ Updates the navigation buttons based on the displayed songs. """
        if (self.__displayedStartIndex + \
                self.__MAXIMUM_DISPLAYED_SONGS) > len(self.__mediaSongs):
            self.__btmpBttnNext.Enable(False)
        elif len(self.__mediaSongs) > self.__MAXIMUM_DISPLAYED_SONGS:
            self.__btmpBttnNext.Enable(True)
        else:
            self.__btmpBttnNext.Enable(False)

        if self.__displayedStartIndex < 1:
            self.__btmpBttnPrevious.Enable(False)
        else:
            self.__btmpBttnPrevious.Enable(True)

        if self.__displayedStartIndex is not None:
            self.__spnCtrlPage.SetRange(1, len(self.__mediaSongs) / \
                self.__MAXIMUM_DISPLAYED_SONGS + 1)
            self.__spnCtrlPage.SetValue(self.__displayedStartIndex / \
                self.__MAXIMUM_DISPLAYED_SONGS + 1)

    def __onPageChanged(self, event):
        """ Hangles the changes in the displayed pages. """
        currentPage = self.__displayedStartIndex / \
                self.__MAXIMUM_DISPLAYED_SONGS + 1
        newPage = self.__spnCtrlPage.GetValue()

        if currentPage > newPage:
            self.__onPreviousPage(event)
        elif currentPage < newPage:
            self.__onNextPage(event)
        else:
            return
