#!/usr/bin/env python
# -*- coding: utf-8 -*-
#******************************************************************************
#**** Copyright (C) 2009, 2010                                             ****
#****   John Schneiderman <JohnMS@member.fsf.org>                          ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import playersignal as SignalBase


class MPlayerSignal(SignalBase.PlayerSignal):
    """ Interface for all player's to process player commands. """

    """
     ATTRIBUTES
    """


    def __init__(self, volumeLevel, arguments):
        """ Default constructor

         Initialises the signals for mplayer.
         int volumeLevel: is the percentage level for playing music.
         string arguments: are any additional arguments to pass along to
           mplayer besides the slave and quiet commands.
        """
        SignalBase.PlayerSignal.__init__(self, volumeLevel, arguments)

    def name(self):
        """ Accessor to the player whose signals it handles. """
        return "MPlayer"

    def play(self, player):
        """ Handles playing a song to the player.

         Starts playing the song stored in self.songFilePath, as well as
           passing along any additional arguements supplied, through the
           player controller.
         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.play(self, player)", self, player
        command = "mplayer -slave -quiet " + self.arguments + self.songFilePath
        player.startPlayer(command)

    def pause(self, player):
        """ Handles pausing a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.pause(self, player)"
        player.writeStandardInput('pause\n')

    def unpause(self, player):
        """ Handles unpausing a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.unpause(self, player)"
        player.writeStandardInput('pause\n')

    def stop(self, player):
        """ Handles stoping a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.stop(self, player)"
        player.writeStandardInput('quit\n')
        player.waitForExit()

    def quit_(self, player):
        """ Handles quiting the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.quit_(self, player)"
        player.writeStandardInput('quit\n')
        player.waitForExit()

    def setVolume(self, player):
        """ Handles volume of a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.setVolume(self, player)"
        player.writeStandardInput('volume %s 1\n' % self.volumeLevel)

    def playerTotalTime(self, player):
        """ Handles total time of a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.totalTime(self, player)"
        player.writeStandardInput('get_time_length\n')
        line = player.readStandardOut()
        while line is not "" and not "ANS_LENGTH" in line:
            line = player.readStandardOut()
        if line:
            self.totalTime = float(line.split("=")[1].replace("\n", ""))

    def playerPositionTime(self, player):
        """ Handles position of a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.playerPositionTime(self, player)"
        player.writeStandardInput('get_time_pos\n')
        line = player.readStandardOut()
        while line is not "" and not "ANS_TIME_POSITION" in line:
            line = player.readStandardOut()
        if line:
            self.positionTime = float(line.split("=")[1].replace("\n", ""))

    def setPlayerPositionTime(self, player):
        """ Handles setting postion of a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.setPlayerPositionTime(self, player)"
        player.writeStandardInput('seek %s 2\n' % self.positionTime)

    def fullScreen(self, player):
        """ Handles playing full screen a song to the player.

         PlayerController player: is the the interface to the OS command line.
        """
        #print " * MPlayerSignal.fullScreen(self, player)"
        player.writeStandardInput('vo_fullscreen 1\n')
