/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef AIBASE_H
#define AIBASE_H

#include "player.h"

class QString;
class CardSequence;
class RuleBase;

/**
 * This is the abstract interface for the way a computer player would play a card in a game.
 * This can be thought of as the "brain" of a player, it holds all the knowledge of each game the AI knows how to play.
 *
 * @author John Schneiderman
 */
class AIBase
{
public:
    /**
     * Default constructor.
     */
    AIBase();
    /**
     * Default destructor.
     */
    virtual ~AIBase();
    /**
     * An accessor to the name of our AI player.
     * @return the name of our AI.
     */
    virtual QString name() const=0;
    /**
     * This selects a number of cards based on the game being played and the strategy the AI uses.
     * @param game is the name of the game or game phase the AI player is currently in. This information should be used to determine what the player needs to do.
     * @param playSequence is the current play sequence of cards in the game.
     * @param rules is the rules for the current game.
     * @param hand is the cards in the AI players hand.
     * @return the cards the player selects.
     */
    virtual CardSequence whichCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const=0;
    /**
     * This gives the skill level for the AI Player for a particular game.
     * @param game is the game we want to know how well the AI player can play.
     * @return is the skill level of the player based on the enumeration in @ref Player .
     */
    virtual Player::skill_Level skillLevel(const QString &game) const=0;
};
#endif
