/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CARD_H
#define CARD_H

#include <iostream>
using std::istream;
using std::ostream;

class QString;
class QTextStream;

/**
 * This is used to represent the properties of a general playing card.
 * @note The current assumption is that the card is a standard US or UK pack.
 *
 * @author John Schneiderman
 * @todo add a function object to allow for the comparisons to be done using a different ordering if needed.
 * @todo separate out the suit and rank information so that we can localise it even more.
 */
class Card
{
public:
    /**
     * These are the names of the available suits for a card.
     * @note that the order of the suits is according to the rules for Bridge.
     * @param SUIT_ERR indicates an error for a suit.
     */
    enum suit_t { SPADES, HEARTS, DIAMONDS, CLUBS, SUIT_ERR };
    /**
     * These are the names of the available ranks for a card.
     * @note that the order of the ranks starts with ace as being low.
     * @param RANK_ERR indicates an error for a rank.
     */
    enum rank_t { ACE, TWO, THREE, FOUR, FIVE, SIX, SEVEN, EIGHT, NINE, TEN, JACK, QUEEN, KING, RANK_ERR };
    /**
     * These are constants used by this class and fellow classes.
     * @param NUMBER_OF_SUITS this is how many suits are in a deck.
     * @param NUMBER_OF_RANKS this is how many ranks are in a deck.
     */
    enum cardConstants { NUMBER_OF_SUITS=4, NUMBER_OF_RANKS=13 };

    /**
     * This constructor creates a card whose rank and suit are RANK_ERR and SUIT_ERR, respectively.
     */
    Card();
    /**
     * This is constructor creates a specific card.
     * @param rank is the rank of the card.
     * @param suit is the suit of the card.
     * @throw KardsGTError if rank or suit are out of scope for their respective types.
     */
    Card(rank_t rank, suit_t suit);
    /**
     * This gives the suit of the card.
     * @return the suit of the card.
     */
    suit_t suit() const;
    /**
     * This gives the rank of the card.
     * @return the rank of the card.
     */
    rank_t rank() const;
    /**
     * This will set our card to a specific card value.
     * @param card is the 2-3 character string representing a card, based on toString().
     * @note the input is not checked for validity, use at your own risk.
     */
    void setCard(const QString &card);
    /**
     * This will set our card to a specific card value.
     * @param rank is the rank for the card.
     * @param suit is the suit for the card.
     * @throw KardsGTError if rank or suit are out of scope for their respective types.
     */
    void setCard(rank_t rank, suit_t suit);
    /**
     * This sets two cards equal to each other.
     * @param card is the card to set *this equal to.
     * @return *this.
     */
    Card& operator=(const Card &card);
    /**
     * This gives a human readable string representation of a card. \n
     * The format of the card is RankSuit. With one character for each, i.e. an ace of spades would be AS, and a nine of clubs would be 9C.
     * @return the formated card.
     */
    QString toString() const;
    /**
     * This determines if a card is empty.
     * @note We assume that a card is empty if it is missing either a suit or a rank.
     * @return true if the card is missing its suit or rank, false elsewise.
     */
    bool isEmpty() const;
    /**
     * This gives the result of adding two card's rank values together.
     * @note This addition is done based on the ordering in rank_t.
     * @param card is the card whose rank is to be added to *this's rank.
     * @return the result of the two cards.
     */
    int operator+(const Card &card);
    /**
     * This gives the result of subtracting two card's rank values from each other.
     * @note This subtraction is done based on the ordering in rank_t.
     * @param card is the card whose rank is to be subtracted from *this's rank.
     * @return the result of the two cards.
     */
    int operator-(const Card &card);
    /**
     * This determines if two cards are equal to each other.
     * @param card is the card to compare to *this.
     * @return true if both the suits and ranks are the same, false elsewise.
     */
    bool operator==(const Card &card) const;
    /**
     * This determines if two cards are not equal to each other.
     * @param card is the card to compare to *this.
     * @return true if both the suits and ranks are not the same, false elsewise.
     */
    bool operator!=(const Card &card) const;
    /**
     * This determines if one card's rank is less than the another.
     * @note This comparison is done based on the ordering in rank_t.
     * @param card is the card to determine if *this is less than.
     * @return true if the rank of *this is less than card, false elsewise.
     */
    bool operator<(const Card &card) const;
    /**
     * This determines if one card's rank is greater than the another.
     * @note This comparison is done based on the ordering in rank_t.
     * @param card is the card to determine if *this is greater than.
     * @return true if the rank of *this is greater than card, false elsewise.
     */
    bool operator>(const Card &card) const;
    /**
     * This determines if one card is less than the other by suit.
     * @note This comparison is done based on the ordering in suit_t.
     * @param card is the card to determine if *this is less than.
     * @return true if the suit of *this is less than card, false elsewise.
     */
    bool lessThan(const Card & card) const;
    /**
     * This is the stream insertion for a card.
     * @param out is the stream to output the card to in the format of RankSuit as described in toString().
     * @param card is the card to output.
     * @return out.
     */
    friend ostream& operator<<(ostream &out, const Card &card);
    /**
     * This is the stream extraction for a card.
     * @param in is the stream to read the card from in the format of RankSuit as described in toString().
     * @param card is the card to read into.
     * @return in.
     */
    friend istream& operator>>(istream &in, Card &card);
    /**
     * This is the stream insertion for a card.
     * @param out is the stream to output the card to in the format of RankSuit as described in toString().
     * @param card is the card to output.
     * @return out.
     */
    friend QTextStream& operator<<(QTextStream &out, const Card &card);
    /**
     * This is the stream extraction for a card.
     * @param in is the stream to read the card from in the format of RankSuit as described in toString().
     * @param card is the card to read into.
     * @return in.
     */
    friend QTextStream& operator>>(QTextStream &in, Card &card);

private:
    /// @param m_rank is the rank of the card.
    rank_t m_rank;
    /// @param m_suit is the suit of the card
    suit_t m_suit;

    /**
     * This generates a human readable representation of a card.
     * @return a string in the format described in toString().
     */
    inline QString generateHumanReadable() const;
    /**
     * This tries to generate the rank for a card.
     * @param rank is the character representing the card's rank.
     * @note that 1 is used for a 10, A is used for an ace, K is used for a king, Q is used for a queen, J is used for a jack, and the rest are all equal to their numerical equivalent.
     * @return the rank found, if the rank cannot be determined, RANK_ERR is returned.
     */
    inline rank_t generateRank(char rank) const;
    /**
     * This tries generate the suit for a card.
     * @param suit is the character representing the card's suit.
     * @note Each of the suits are identified by the first letter that they start with.
     * @return the suit found, if the suit cannot be determined, SUIT_ERR is returned.
     */
    inline suit_t generateSuit(char suit) const;
};
#endif
