/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CARD_PROPERTIES_H
#define CARD_PROPERTIES_H

#include "cardsequence.h"
#include "card.h"

#include <vector>
using std::vector;
#include <string>
using std::string;

class RuleBase;

/**
 * This is designed to determine the unique properties of a sequence of cards.
 *
 * @author John Schneiderman
 */
class CardProperties
{
public:
    /**
     * Default constructor.
     */
    CardProperties();
    /**
     * This constructor is used to start off with a card sequence.
     * @param sequence is the card sequence you wish to determine the properties of.
     */
    CardProperties(const CardSequence &sequence);
    /**
     * This gives you the sequence that we're examining.
     * @return the card sequence that we're examining.
     */
    const CardSequence& sequence() const;
    /**
     * This sets the sequence of cards you want to examine.
     * @param sequence is the sequence of cards you want to examine.
     */
    void setSequence(const CardSequence &sequence);
    /**
     * This goes through our sequence and finds all singletons.
     * @return all singletons found in the sequence of cards.
     * @todo implement me!
     */
    vector<Card> singletons() const;
    /**
     * This goes through our sequence and finds all doubletons.
     * @return all doubletons found in the sequence of cards.
     * @todo implement me!
     */
    vector<CardSequence> doubletons() const;
    /**
     * This goes through our sequence and finds all pairs.
     * @return each unique pair that is found in the sequence of cards.
     */
    vector<CardSequence> pairs() const;
    /**
     * This goes through our sequence and finds all the unique runs.
     * @return the longest unique runs found in the sequence of cards.
     */
    vector<CardSequence> runs() const;
    /**
     * This goes through our sequence and finds all the unique flushes.
     * @return the largest unique flushes that can be found in the sequence of cards.
     */
    vector<CardSequence> flushes() const;
    /**
     * This goes through our sequence and finds all combinations whose sum equals a specified value.
     * @param goal is the value you aspire to combine to using the sequence of cards.
     * @param rules is the specific rules for the card game being played.
     * @return the unique combinations that sum up to goal.
     */
    vector<CardSequence> summations(int goal, const RuleBase &rules) const;
    /**
     * This goes through our sequence and finds all the cards of a specific suit.
     * @param suit is the suit we're looking for.
     * @return all the suits that match the suit we want.
     */
    CardSequence suits(Card::suit_t suit) const;
    /**
     * This goes through our sequence and finds all the cards of a specific rank.
     * @param rank is the rank we're looking for.
     * @return all the suits that match the rank we want.
     */
    CardSequence ranks(Card::rank_t rank) const;


protected:
    /**
     * These are the rules used to determine what patterns constitutes something in a sequence of cards.
     * @param SINGLETON is when in the sequence you only have one card in a suit.
     * @param DOUBLETON is when in the sequence you only have two cards in a suit.
     * @param PAIR is when in the sequence you have cards of the same rank.
     * @param RUN is when in the sequence you have cards increasing in sequence by one.
     * @param FLUSH is when in the sequence you have cards of the same suit.
     */
    enum propertyRules { SINGLETON=1, DOUBLETON=2, PAIR=2, RUN=3, FLUSH=4 };

    /**
     * This goes through the card sequence and arranges them according to a binary ordering. \n
     * For example: QS, KS, JS would return the following combinations: \n
     * JS \n
     * QS \n
     * QS JS \n
     * KS \n
     * KS JS \n
     * KS QS \n
     * KS QS JS
     * @param seq is the card sequence to parse and expand, generating the combinations.
     * @return an array filled with all possible unique combinations for the card sequence.
     */
    vector<CardSequence> parseCardSequence(const CardSequence &seq) const;
    /**
     * Searches through and finds all unique sequences.
     * @note A unique sequence is a sequence that has a card that is not in any of the other sequences.
     * @param examineSequence is the sequences to examine.
     * @return the unique sequences found.
     */
    vector<CardSequence> uniqueSequences(const vector<CardSequence> &examineSequence) const;

private:
    /// @param m_cardSequences is the sequence of card combinations.
    vector<CardSequence> m_cardSequences;
};
#endif
