/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef KARD_H
#define KARD_H

#include "card.h"

#include <qwidget.h>
#include <qpixmap.h>
#include <qpoint.h>
#include <qdatastream.h>

class QSize;
class KardDrag;

/**
 * This draws a card image. This is not particularly useful directly.
 *
 * @author John Schneiderman
 */
class Kard: public QWidget
{
    Q_OBJECT
    Q_PROPERTY(QPixmap m_cardBack READ cardBack WRITE setCardBack)
    Q_PROPERTY(QPixmap m_cardFront READ cardFront WRITE setCardFront)
    Q_PROPERTY(bool m_faceUp READ isFaceUp WRITE setFaceUp)
    Q_PROPERTY(bool m_selected READ isSelected WRITE setSelected)

public:
    /**
     * These are the preferred sizes.
     * @param PREFERRED_WIDTH is the preferred width of the card.
     * @param PREFERRED_HEIGHT is the preferred height of the card.
     */
    enum widgetSize { PREFERRED_WIDTH=60, PREFERRED_HEIGHT=90 };

    /**
     * A constructor to create a plain card.
     * @param parent is the parent of this widget.
     * @param name is the name of this widget.
     */
    Kard(QWidget *parent=0, const char *name=0);
    /**
     * A constructor to create a specific card.
     * @param card is the card the widget represents.
     * @param faceUp is weather the widget should display the card face up or face down. True means face up.
     * @param parent is the parent of this widget.
     * @param name is the name of this widget.
     */
    Kard(const Card &card, bool faceUp, QWidget *parent=0, const char *name=0);
    /**
     * Destructor for our object..
     */
    ~Kard();
    /**
     * This gives the preferred size of the card.
     * @return the size of the card.
     */
    QSize sizeHint() const;
    /**
     * Gives the current image for the back of the card card.
     * @return the card's back image.
     */
    const QPixmap& cardBack() const;
    /**
     * Gives the current image for the front of the card.
     * @return the card's front image.
     */
    const QPixmap& cardFront() const;
    /**
     * This determines if the card is facing up.
     * @return true if it is facing up, false elsewise.
     */
    bool isFaceUp() const;
    /**
     * This determines if a user has selected this card.
     * @return true if the card has been selected, false elsewise.
     */
    bool isSelected() const;
    /**
     * This gives what card is currently being displayed.
     * @return the card displayed.
     */
    const Card& card();
    /**
     * This will set the card's back image. Generally this will be some design pattern.
     * @param cardBack is the image that the back of the card should have.
     * @throw KardsGTError if back is null.
     */
    void setCardBack(const QPixmap &cardBack);
    /**
     * This will set the card's front image. Generally this will be the suit and rank of a card.
     * @param cardFront is the image that the front of the card should have.
     * @throw KardsGTError if front is null.
     */
    void setCardFront(const QPixmap &cardFront);
    /**
     * This sets the card we want to be displayed. The image itself is pulled from the mime source in the binary.
     * @param card is the card to display.
     * @throw KardsGTError if the card image is not found.
     */
    void setCard(const Card &card);
    /**
     * This sets whether a card should be facing up.
     * @param faceUp should be true if you wish for the front image to be displayed, false elsewise.
     */
    void setFaceUp(bool faceUp);
    /**
     * This sets whether a card has been selected.
     * @param select when true will select this kard, false will un-set our selection.
     * @note if select is true then cardSelected signal will be emitted.
     * @note if the card had already been selected then nothing will happen.
     */
    void setSelected(bool select);
    /**
     * This flips the displayed images. Front to back, back to front.
     */
    void flipCard();
    /**
     * This will assign a Kard into *this.
     * @param kard is the kard we want to assign into *this.
     * @return *this.
     */
    Kard& operator=(const Kard &kard);

signals:
    /**
     * This is the signal emitted when the card is selected by a double left click.
     * @param kard is *this.
     */
    void cardSelected(Kard &kard);

protected:
    /**
     * This draws the card image.
     * @param event is the source of the generating event.
     */
    virtual void paintEvent(QPaintEvent *event);
    /**
     * This is how a user selects a card.
     * @param event is the generating event.
     * @note the meaning of each click is as follows: \n
     * Double clickling the left mouse button will select the card.
     */
    virtual void mouseDoubleClickEvent(QMouseEvent *event);
    /**
     * This starts the dragging check for the card.
     * @param event is the generating event.
     */
    virtual void mousePressEvent(QMouseEvent *event);
    /**
     * This checks to see if we are trying to drag the Kard.
     * @param event is the generating event.
     */
    virtual void mouseMoveEvent(QMouseEvent *event);
    /**
     * This gives the information for the type of drag event.
     * @param event is the generating event.
     * @note our event is defined in @ref KardDrag .
     */
    virtual void dragEnterEvent(QDragEnterEvent *event);

private:
    /**
     * These are the constants we use for the kards.
     * @param START_DRAG_DISTANCE is the starting dragging distance.
     */
    enum constants { START_DRAG_DISTANCE=4 };

    /// @param m_cardBack is the image on the back of the card.
    QPixmap m_cardBack;
    /// @param m_cardFront is the image on the front of the card.
    QPixmap m_cardFront;
    /// @param m_faceUp is true if we are displaying the front of the card and false if we're displaying the back of the card.
    bool m_faceUp;
    /// @param m_selected is true if the card has been selected, false elsewise.
    bool m_selected;
    /// @param m_card is the card that we are displaying to the user.
    Card m_card;
    /// @param m_dragPos is the position where our drag event began.
    QPoint m_dragPos;

    /**
     * This launches our drag event.
     */
    void startDrag();
    /**
     * This draws the kard image.
     * @param painter is where to paint the kard.
     */
    void draw(QPainter &painter);
};
#endif
