/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardplayer.h"
#include "kardsgterror.h"

#include <qimage.h>
#include <qevent.h>
#include <qpainter.h>
#include <qpoint.h>
#include <qfont.h>
#include <qbrush.h>

KardPlayer::KardPlayer(QWidget *parent, const char *name):QWidget(parent, name, WPaintClever | WRepaintNoErase), Player(), m_playerImage(), m_caption()
{
    m_captionTimerId=0;
}

QSize KardPlayer::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const QString& KardPlayer::caption() const
{
    return m_caption;
}

void KardPlayer::setCaption(QString caption)
{
    if (m_caption.isEmpty())
        m_caption=caption;
    else
        m_caption=caption + m_caption;
    m_captionTimerId=startTimer(CAPTION_DISPLAY_TIME);
    update();
    updateGeometry();
}

void KardPlayer::clear()
{
    m_caption="";
    update();
    updateGeometry();
}

QPixmap KardPlayer::playerImage() const
{
    return m_playerImage;
}

void KardPlayer::setPlayerImage(const QPixmap &playerImage)
{
    if (playerImage.isNull())
        throw KardsGTError("KardPlayer", "setPlayerImage", "playerImage is null!");
    else
        m_playerImage=playerImage;
    m_playerImage.detach();
    update();
    updateGeometry();
}

void KardPlayer::setPlayerImage(QString imageFilename)
{
    m_playerImage=QImage::fromMimeSource(imageFilename.lower() + ".png");
    if (m_playerImage.isNull())
        throw KardsGTError(QString("KardPlayer"), QString("setPlayerImage"), QString("%1 was not found.").arg(imageFilename));
    m_playerImage.detach();
    update();
    updateGeometry();
}

void KardPlayer::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect=event->rect();

    QSize newSize=rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    painter.translate(-rect.x(), -rect.y());
    draw(painter);

    // Draw creation
    bitBlt(this, rect.x(), rect.y(), &pixmap, 0, 0, rect.width(), rect.height());
}

void KardPlayer::draw(QPainter &painter)
{
    const int FONT_SIZE=10; // The font size we want to use.
    const int FONT_WIDTH=9; // The scale factor needed to make 1 char = 1 L.U..
    const int CHARS_PER_LINE=3; // How many characters should appear on a single line.

    painter.drawPixmap(painter.window(), m_playerImage);
    painter.setFont(QFont("courier", FONT_SIZE));
    if (! m_caption.isEmpty())
    {
        int textX=painter.window().width() / 2;
        int textY=painter.window().height() / 4;
        int windowWidth=painter.window().width();
        int width, height;
        bool textWillFit=true;

        // Find the width of our caption
        if ((static_cast<int>(m_caption.length()) * FONT_WIDTH) < windowWidth)
            width=m_caption.length() * FONT_WIDTH;
        else
        {
            width=windowWidth - textX + 5;
            textWillFit=false;
        }

        // Find the height of our caption
        if (textWillFit)
            height = FONT_SIZE + 6;
        else
            height = ((static_cast<int>(m_caption.length()) * FONT_WIDTH) / 3) * 2;

        // Create the caption bubble
        painter.setBackgroundMode(Qt::TransparentMode);
        painter.setBrush(Qt::white);
        painter.drawEllipse(textX - 4, textY - 4, width, height);
        painter.drawRect(textX - 3, textY + 5, 3, 5);

        // Draw the caption
        painter.setPen(Qt::black);
        if (textWillFit)
            painter.drawText(textX, textY + 8, m_caption);
        else
        {
            for (int i=0; i < static_cast<int>(m_caption.length()); i+=CHARS_PER_LINE)
                painter.drawText(textX, textY + (i * 5) + 12, m_caption.mid(i, CHARS_PER_LINE));
        }
    }
    painter.setPen(Qt::red);
    painter.setBackgroundColor(Qt::white);
    painter.setBackgroundMode(Qt::OpaqueMode);
    // Show the player if it's their turn, they're the dealer, else show them the skill level.
    if (m_turn)
        painter.drawText(2, 10, "T"); // 2, 10 should be the upper-left corner.
    else if (m_dealer)
        painter.drawText(2, 10, "D");
    else
        painter.drawText(2, 10, QString("%1").arg(m_level));
}

void KardPlayer::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == m_captionTimerId)
    {
        killTimer(m_captionTimerId);
        m_captionTimerId=0;
        clear();
    }
    else
        QWidget::timerEvent(event);
}
