/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef KARDPLAYER_H
#define KARDPLAYER_H

#include "player.h"

#include <qwidget.h>
#include <qpixmap.h>
#include <qstring.h>

class QPainter;

/**
 * This is the graphical representation of a player.
 *
 * @author John Schneiderman
 * @todo make the caption bubble more intelligent, i.e. know how to break up words better, etc.
 */
class KardPlayer: public QWidget, public Player
{
    Q_OBJECT
    Q_PROPERTY(QPixmap m_playerImage READ playerImage WRITE setPlayerImage)
    Q_PROPERTY(QString m_caption READ caption WRITE setCaption)

public:
    /**
     * KardPlayer constructor
     * @param parent is the parent window.
     * @param name is the name of this object.
     */
    KardPlayer(QWidget *parent = 0, const char *name = 0);
    /**
     * This gives the preferred size of the player's image.
     * @return the size of the player's image.
     */
    QSize sizeHint() const;
    /**
     * This is the accessor for the current player's image.
     */
    QPixmap playerImage() const;
    /**
     * This sets the image of the player.
     * @param playerImage is what to set the player's image to.
     * @throw KardsGTError if the playerImage is null.
     */
    void setPlayerImage(const QPixmap &playerImage);
    /**
     * This sets the image of the player.
     * @param imageFilename is the name of the file for the player.
     * @note We assume the file is named imageFilename.png and it resides inside the binary.
     * @throw KardsGTError if the image file cannot be found.
     */
    void setPlayerImage(QString imageFilename);
    /**
     * This is the accessor to what the player "said".
     */
    const QString& caption() const;
    /**
     * This causes the player to "say" something.
     * @param caption is what the player says.
     * @note if we're still showing a previous caption, we'll combine the messages.
     */
    void setCaption(QString caption);
    /**
     * This clears what the player has said.
     */
    void clear();

protected:
    /**
     * This draws the players image.
     * @param event is the generating event.
     */
    virtual void paintEvent(QPaintEvent *event);
    /**
     * This will erase the caption.
     * @param event is the timer event to clear our caption.
     */
    virtual void timerEvent(QTimerEvent *event);

private:
    /**
     * These are the constants that are used in this class.
     * @param PREFERRED_WIDTH is the preferred width of a player's image.
     * @param PREFERRED_HEIGHT is the preferred height of a player's image.
     * @param CAPTION_DISPLAY_TIME is how many ms to display the caption.
     */
    enum constants { PREFERRED_WIDTH=25, PREFERRED_HEIGHT=50, CAPTION_DISPLAY_TIME=5000 };

    /// @param m_playerImage is the players image.
    QPixmap m_playerImage;
    /// @param m_caption is the caption for the player.
    QString m_caption;
    /// @param m_captionTimerId is the identification for the caption timer.
    int m_captionTimerId;

    /**
     * This draws our player and anything he says.
     * @param painter is what we want to paint on.
     */
    void draw(QPainter &painter);
};
#endif
