/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardsequence.h"
#include "kard.h"
#include "kardsgterror.h"
#include "karddrag.h"
#include "card.h"
#include "kardpile.h"

#include <qlayout.h>
#include <qpainter.h>
#include <qevent.h>
#include <iterator>

KardSequence::KardSequence(QWidget *parent, const char * name, bool vertical):QWidget(parent, name, WPaintClever | WRepaintNoErase), m_cardSequence(), m_kards()
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_vertical=vertical;
    if (m_vertical)
        m_pLayout=new QVBoxLayout(this, MARGIN, HORIZONTAL, "KardSequenceLayout");
    else
        m_pLayout=new QHBoxLayout(this, MARGIN, HORIZONTAL, "KardSequenceLayout");
    m_faceUp = false;
    setAcceptDrops(true);
}

KardSequence::~KardSequence()
{
    int size=static_cast<int>(m_kards.size());

    for (int i=0; i < size; ++i)
        delete m_kards[i];
    delete m_pLayout;
}

QSize KardSequence::sizeHint()
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const CardSequence& KardSequence::cardSequence() const
{
    return m_cardSequence;
}

bool KardSequence::isFaceUp() const
{
    return m_faceUp;
}

bool KardSequence::isVertical() const
{
    return m_vertical;
}

int KardSequence::size() const
{
    return m_cardSequence.size();
}

void KardSequence::setFaceUp(bool faceUp)
{
    m_faceUp = faceUp;
    update();
    updateGeometry();
}

void KardSequence::setVertical(bool vertical)
{
    m_vertical = vertical;
    update();
    updateGeometry();
}

void KardSequence::clear()
{
    int size=m_kards.size();

    for (int i=0; i < size; ++i)
        removeKard(*m_kards[0]);
    m_kards.clear();
    m_cardSequence.clear();
    delete m_pLayout;
    if (m_vertical)
        m_pLayout=new QVBoxLayout(this, MARGIN, HORIZONTAL, "KardSequenceLayout");
    else
        m_pLayout=new QHBoxLayout(this, MARGIN, HORIZONTAL, "KardSequenceLayout");
    update();
    updateGeometry();
}

void KardSequence::setCardSequence(const CardSequence &sequence)
{
    // Don't bother if we're already the same.
    if (m_cardSequence == sequence)
        return;

    //Set up the new cards
    clear();
    for (int i=0; i < sequence.size(); ++i)
        addCard(sequence[i]);
    update();
    updateGeometry();
}

void KardSequence::addKard(Kard &kard)
{
    addCard(kard.card());
}

void KardSequence::addCard(const Card &card)
{
    Kard *kard = new Kard(card, m_faceUp, this, card.toString());

    m_cardSequence.addCard(card);
    connect(kard, SIGNAL(cardSelected(Kard &)), this, SIGNAL(kardSelected(Kard &)));
    m_pLayout->addWidget(kard);
    m_kards.push_back(kard);
    update();
    updateGeometry();
}

void KardSequence::removeKard(Kard &kard)
{
    removeCard(kard.card());
}

void KardSequence::removeCard(const Card &card)
{
    vector<Kard *>::iterator iter=m_kards.begin();
    int i=0;

    while (iter != m_kards.end())
    {
        if ((*iter)->card() == card)
        {
            (*iter)->hide();
            m_pLayout->remove(*iter);
            m_cardSequence.removeCard(card);
            //             delete m_kards[i];
            //             m_kards[i]=0;
            m_kards.erase(iter);
            update();
            updateGeometry();
            return;
        }
        i++, iter++;
    }
    throw KardsGTError(QString("KardSequence"), QString("removeCard"), "Failed to remove " + card.toString() + ".");
}

void KardSequence::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect=event->rect();

    QSize newSize=rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    painter.translate(-rect.x(), -rect.y());
    draw(painter);
    bitBlt(this, rect.x(), rect.y(), &pixmap, 0, 0, rect.width(), rect.height());
}

void KardSequence::draw(QPainter &painter)
{
    painter.drawRect(painter.window());
    for (int i=0; i < m_cardSequence.size(); ++i)
    {
        m_kards[i]->setFaceUp(m_faceUp);
        m_kards[i]->show();
    }
}

void KardSequence::setSelected(const Card &card)
{
    vector<Kard *>::iterator iter=m_kards.begin();

    while (iter != m_kards.end())
    {
        if ((*iter)->card() == card)
        {
            (*iter)->setSelected(true);
            break;
        }
        iter++;
    }
}

void KardSequence::dragEnterEvent(QDragEnterEvent *event)
{
    event->accept(event->provides("application/x-kard"));
}

void KardSequence::dropEvent(QDropEvent *event)
{
    Kard kard;

    if (KardDrag::decode(event, kard))
    {
        QWidget *fromWidget = event->source();

        if (fromWidget && (fromWidget != this) && fromWidget->inherits("KardSequence"))
        {
            KardSequence *fromKardSequence = static_cast<KardSequence *>(fromWidget);
            fromKardSequence->setSelected(kard.card());
        }
        else if (fromWidget && (fromWidget == this) && fromWidget->inherits("KardSequence"))
        {
            /// @todo add sorting code here.
        }
        else if (fromWidget && (fromWidget != this) && fromWidget->inherits("KardPile"))
        {
            KardPile *fromKardPile = static_cast<KardPile *>(fromWidget);
            fromKardPile->setSelected(kard.card());
        }

    }
}
