/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "cribbageboard.h"
#include "cribbageblock.h"
#include "cribbagepeg.h"

#include <qpainter.h>
#include <qpixmap.h>
#include <qrect.h>
#include <qbrush.h>
#include <qstring.h>

CribbageBoard::CribbageBoard(QWidget *parent, const char *name):QWidget(parent, name, WPaintClever | WRepaintNoErase)
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    setBackgroundColor(QColor(210, 180, 140)); //A light brown.

    //Create our pegs.
    for (int i=0; i < NUMBER_OF_PEGS; ++i)
    {
        m_redPegs.push_back(new CribbagePeg(red, QPoint(ZERO_ZONE_X + 1, ZERO_ZONE_Y + (i * 2) + 1), (i == 0) ? true : false)); // i == 0 then it is a lead CribbagePeg
        m_bluePegs.push_back(new CribbagePeg(blue, QPoint(ZERO_ZONE_X + 4, ZERO_ZONE_Y + (i * 2) + 1), (i == 0) ? true : false)); // i == 0 then it is a lead CribbagePeg
    }
    m_legendPegs.push_back(pair<CribbagePeg *, QString>(new CribbagePeg(red, QPoint(ZERO_ZONE_X + 1, ZERO_ZONE_Y + 30), true), "Red Player"));
    m_legendPegs.push_back(pair<CribbagePeg *, QString>(new CribbagePeg(blue, QPoint(ZERO_ZONE_X + 1, ZERO_ZONE_Y + 35), true), "Blue Player"));

    //Create our blocks
    for (int i=0; i < NUMBER_VERTICAL_BLOCKS; ++i)
    {
        m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_PEGS_PER_BLOCK, FIRST_ZONE_X, FIRST_ZONE_Y + (i * 10)));
        m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_PEGS_PER_BLOCK, THIRD_ZONE_X, THIRD_ZONE_Y + (i * 10)));
        m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_PEGS_PER_BLOCK, FIFTH_ZONE_X, FIFTH_ZONE_Y + (i * 10)));
        m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_PEGS_PER_BLOCK, FIRST_ZONE_X + BLOCK_OFFSET, FIRST_ZONE_Y + (i * 10)));
        m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_PEGS_PER_BLOCK, THIRD_ZONE_X - BLOCK_OFFSET, THIRD_ZONE_Y + (i * 10)));
        m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_PEGS_PER_BLOCK, FIFTH_ZONE_X + BLOCK_OFFSET, FIFTH_ZONE_Y + (i * 10)));
    }
    m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_PEGS_PER_BLOCK, SECOND_ZONE_X, SECOND_ZONE_Y, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_PEGS_PER_BLOCK, SECOND_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 2), SECOND_ZONE_Y, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_PEGS_PER_BLOCK, FOURTH_ZONE_X, FOURTH_ZONE_Y, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_OF_PEGS, ZERO_ZONE_X, ZERO_ZONE_Y));
    m_blocks.push_back(new CribbageBlock(darkRed, NUMBER_OF_STAKES, STAKE_ZONE_X, STAKE_ZONE_Y, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_PEGS_PER_BLOCK, SECOND_ZONE_X, SECOND_ZONE_Y + BLOCK_OFFSET, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_PEGS_PER_BLOCK, SECOND_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 2), SECOND_ZONE_Y + BLOCK_OFFSET, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_PEGS_PER_BLOCK, FOURTH_ZONE_X, FOURTH_ZONE_Y - BLOCK_OFFSET, CribbageBlock::HORIZONTAL));
    m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_OF_PEGS, ZERO_ZONE_X + BLOCK_OFFSET, ZERO_ZONE_Y));
    m_blocks.push_back(new CribbageBlock(darkBlue, NUMBER_OF_STAKES, STAKE_ZONE_X, STAKE_ZONE_Y + BLOCK_OFFSET, CribbageBlock::HORIZONTAL));

    // This is the first pegging zone 0-36
    m_peggingsZones.push_back(new QPoint(FIRST_ZONE_X, FIRST_ZONE_Y + 69));
    // This is the second pegging zone 36-46
    m_peggingsZones.push_back(new QPoint(SECOND_ZONE_X, SECOND_ZONE_Y + 1));
    // This is the third pegging zone 46-81
    m_peggingsZones.push_back(new QPoint(THIRD_ZONE_X + 1, THIRD_ZONE_Y + 1));
    // This is the fourth zone 81-86
    m_peggingsZones.push_back(new QPoint(FOURTH_ZONE_X + 8, FOURTH_ZONE_Y + 1));
    // This is the fifth zone 86-121
    m_peggingsZones.push_back(new QPoint(FIFTH_ZONE_X + 1, FIFTH_ZONE_Y + 69));
    // This is the sixth zone (Stakes won) 121
    m_peggingsZones.push_back(new QPoint(SIXTH_ZONE_X + 1, SIXTH_ZONE_Y + 1));
    // This is the stake zone
    m_peggingsZones.push_back(new QPoint(STAKE_ZONE_X, STAKE_ZONE_Y + 1));
}

CribbageBoard::~CribbageBoard()
{
    int size=static_cast<int>(m_redPegs.size());

    for (int i=0; i < size; ++i)
        delete m_redPegs[i];
    size=static_cast<int>(m_bluePegs.size());
    for (int i=0; i < size; ++i)
        delete m_bluePegs[i];
    size=static_cast<int>(m_peggingsZones.size());
    for (int i=0; i < size; ++i)
        delete m_peggingsZones[i];
    size=static_cast<int>(m_blocks.size());
    for (int i=0; i < size; ++i)
        delete m_blocks[i];
    size=static_cast<int>(m_legendPegs.size());
    for (int i=0; i < size; ++i)
        delete m_legendPegs[i].first;
}

pair<int, int> CribbageBoard::scores(const QColor &pegColour) const
{
    pair<int, int> playerScores;

    if (pegColour == red)
    {
        if (m_redPegs.front()->isFront())
        {
            playerScores.first = m_redPegs[0]->score();
            playerScores.second = m_redPegs[1]->score();
        }
        else
        {
            playerScores.first = m_redPegs[1]->score();
            playerScores.second = m_redPegs[0]->score();
        }
    }
    else if (pegColour == blue)
    {
        if (m_bluePegs.front()->isFront())
        {
            playerScores.first = m_bluePegs[0]->score();
            playerScores.second = m_bluePegs[1]->score();
        }
        else
        {
            playerScores.first = m_bluePegs[1]->score();
            playerScores.second = m_bluePegs[0]->score();
        }
    }
    return playerScores;
}

void CribbageBoard::setScores(const QColor &pegColour, pair<int, int> scores)
{
    if (pegColour == red)
    {
        m_redPegs[0]->setScore(scores.first);
        m_redPegs[1]->setScore(scores.second);
    }
    else if (pegColour == blue)
    {
        m_bluePegs[0]->setScore(scores.first);
        m_bluePegs[1]->setScore(scores.second);
    }
}

void CribbageBoard::pegScore(const QColor &pegColour, int score)
{
    int leadPeg=0, nonLeadPeg=0;

    if ((pegColour == m_redPegs.front()->colour()))
    {
        // Find leading CribbagePeg
        if (m_redPegs.front()->isFront())
        {
            leadPeg=0;
            nonLeadPeg=1;
        }
        else
        {
            leadPeg=1;
            nonLeadPeg=0;
        }

        // Return if the score hasn't changed.
        if (score == m_redPegs[leadPeg]->score())
            return;

        // If we're starting out, move the first CribbagePeg to the leading position
        if (m_redPegs[leadPeg]->score() == 0)
        {
            movePeg(*m_redPegs[leadPeg], score);
            m_redPegs[leadPeg]->setScore(score);
        }
        // Move the non leading CribbagePeg to the leading new score position.
        else if (m_redPegs[leadPeg]->score() < SIXTH_ZONE_ENDING)
        {
            movePeg(*m_redPegs[nonLeadPeg], score);
            m_redPegs[nonLeadPeg]->setScore(score);

            // Reset who's in the front
            m_redPegs[leadPeg]->setLead(false);
            m_redPegs[nonLeadPeg]->setLead(true);
        }
        else
        {
            movePeg(*m_redPegs[nonLeadPeg], FIFTH_ZONE_ENDING);
            m_redPegs[nonLeadPeg]->setScore(FIFTH_ZONE_ENDING);

            // Reset who's in the front
            m_redPegs[leadPeg]->setLead(false);
            m_redPegs[nonLeadPeg]->setLead(true);
        }

    }
    else if (pegColour == m_bluePegs.front()->colour())
    {
        // Find leading CribbagePeg
        if (m_bluePegs.front()->isFront())
        {
            leadPeg=0;
            nonLeadPeg=1;
        }
        else
        {
            leadPeg=1;
            nonLeadPeg=0;
        }

        // Return if the score hasn't changed.
        if (score == m_bluePegs[leadPeg]->score())
            return;

        // If we're starting out, move the first CribbagePeg to the leading position
        if (m_bluePegs[leadPeg]->score() == 0)
        {
            movePeg(*m_bluePegs[leadPeg], score);
            m_bluePegs[leadPeg]->setScore(score);
        }
        // Move the non leading CribbagePeg to the leading new score position.
        else if (m_bluePegs[leadPeg]->score() < SIXTH_ZONE_ENDING)
        {
            movePeg(*m_bluePegs[nonLeadPeg], score);
            m_bluePegs[nonLeadPeg]->setScore(score);

            // Reset who's in the front
            m_bluePegs[leadPeg]->setLead(false);
            m_bluePegs[nonLeadPeg]->setLead(true);
        }
        else
        {
            movePeg(*m_bluePegs[nonLeadPeg], FIFTH_ZONE_ENDING);
            m_bluePegs[nonLeadPeg]->setScore(FIFTH_ZONE_ENDING);

            // Reset who's in the front
            m_bluePegs[leadPeg]->setLead(false);
            m_bluePegs[nonLeadPeg]->setLead(true);
        }
    }
}

void CribbageBoard::movePeg(CribbagePeg &CribbagePeg, int newScore)
{
    // Are we in the first zone
    if (newScore < FIRST_ZONE_ENDING)
        CribbagePeg.setPosition(CribbagePeg.x(), m_peggingsZones[0]->y() - (newScore * 2) + 2);
    // Are we in the second zone
    else if (newScore < SECOND_ZONE_ENDING)
        if (CribbagePeg.colour() == red)
            CribbagePeg.setPosition(m_peggingsZones[1]->x() + ((newScore - FIRST_ZONE_ENDING) * 2), m_peggingsZones[1]->y());
        else
            CribbagePeg.setPosition(m_peggingsZones[1]->x() + ((newScore - FIRST_ZONE_ENDING) * 2), m_peggingsZones[1]->y() + 3);
    // Are we in the third zone
    else if (newScore < THIRD_ZONE_ENDING)
        if (CribbagePeg.colour() == red)
            CribbagePeg.setPosition(m_peggingsZones[2]->x(), m_peggingsZones[2]->y() + ((newScore - 46) * 2));
        else
            CribbagePeg.setPosition(m_peggingsZones[2]->x() - 3, m_peggingsZones[2]->y() + ((newScore - 46) * 2));
    // Are we in the fourth zone
    else if (newScore < FOURTH_ZONE_ENDING)
        if (CribbagePeg.colour() == red)
            CribbagePeg.setPosition(m_peggingsZones[3]->x() - ((newScore - 81) * 2), m_peggingsZones[3]->y());
        else
            CribbagePeg.setPosition(m_peggingsZones[3]->x() - ((newScore - 81) * 2), m_peggingsZones[3]->y() - 3);
    // Are we in the fifth zone
    else if (newScore < FIFTH_ZONE_ENDING)
        if (CribbagePeg.colour() == red)
            CribbagePeg.setPosition(m_peggingsZones[4]->x(), m_peggingsZones[4]->y() - ((newScore - 86) * 2));
        else
            CribbagePeg.setPosition(m_peggingsZones[4]->x() + 3, m_peggingsZones[4]->y() - ((newScore - 86) * 2));
    else
        CribbagePeg.setPosition(SIXTH_ZONE_X, SIXTH_ZONE_Y);
    CribbagePeg.setScore(newScore);
    update();
    updateGeometry();
}

void CribbageBoard::pegStakes(const QColor &pegColour, int stakes)
{
    if (stakes > NUMBER_OF_STAKES)
        return;

    if ((pegColour == m_redPegs.front()->colour()))
    {
        if (m_redPegs.back()->score() == stakes)
            return;
        else
            m_redPegs.back()->setPosition(m_peggingsZones.back()->x() + ((stakes - 1) * 2), m_peggingsZones.back()->y());
        m_redPegs.back()->setScore(stakes);
    }
    else if ((pegColour == m_bluePegs.front()->colour()))
    {
        if (m_bluePegs.back()->score() == stakes)
            return;
        else
            m_bluePegs.back()->setPosition(m_peggingsZones.back()->x() + ((stakes - 1) * 2), m_peggingsZones.back()->y() + 3);
        m_bluePegs.back()->setScore(stakes);
    }
    update();
    updateGeometry();
}

void CribbageBoard::reset()
{
    int size = NUMBER_OF_PEGS - 1; // The last CribbagePeg is assumed to be the stake CribbagePeg.

    for (int i=0; i < size; ++i)
        m_redPegs[i]->reset();
    for (int i=0; i < size; ++i)
        m_bluePegs[i]->reset();
    update();
    updateGeometry();
}

void CribbageBoard::clear()
{
    int size = NUMBER_OF_PEGS;

    for (int i=0; i < size; ++i)
        m_redPegs[i]->reset();
    for (int i=0; i < size; ++i)
        m_bluePegs[i]->reset();
    update();
    updateGeometry();
}

void CribbageBoard::setName(const QColor &pegColour, const QString &name)
{
    int size=m_legendPegs.size();

    for (int i=0; i < size; ++i)
        if (m_legendPegs[i].first->colour() == pegColour)
        {
            m_legendPegs[i].second =  name;
            break;
        }
    update();
    updateGeometry();
}

QSize CribbageBoard::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

void CribbageBoard::draw(QPainter *painter)
{
    const int TEXT_OFFSET=3;

    // Draw the blocks
    for (int i=0; i < static_cast<int>(m_blocks.size()); ++i)
        m_blocks[i]->draw(painter);

    // Draw the lines connecting each CribbageBlock
    painter->setPen(QPen(darkRed));
    painter->drawLine(FIRST_ZONE_X, FIRST_ZONE_Y, SECOND_ZONE_X, SECOND_ZONE_Y);
    painter->drawLine(SECOND_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 4), SECOND_ZONE_Y, THIRD_ZONE_X, THIRD_ZONE_Y);
    painter->drawLine(THIRD_ZONE_X, THIRD_ZONE_Y + (NUMBER_PEGS_PER_BLOCK * NUMBER_VERTICAL_BLOCKS * 2), FOURTH_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 2), FOURTH_ZONE_Y);
    painter->drawLine(FOURTH_ZONE_X, FOURTH_ZONE_Y, FIFTH_ZONE_X, FIFTH_ZONE_Y + (NUMBER_PEGS_PER_BLOCK * NUMBER_VERTICAL_BLOCKS * 2));
    painter->setPen(QPen(darkBlue));
    painter->drawLine(FIRST_ZONE_X + BLOCK_OFFSET, FIRST_ZONE_Y, SECOND_ZONE_X, SECOND_ZONE_Y + BLOCK_OFFSET);
    painter->drawLine(SECOND_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 4), SECOND_ZONE_Y + BLOCK_OFFSET, THIRD_ZONE_X - BLOCK_OFFSET, THIRD_ZONE_Y);
    painter->drawLine(THIRD_ZONE_X - BLOCK_OFFSET, THIRD_ZONE_Y  + (NUMBER_PEGS_PER_BLOCK * NUMBER_VERTICAL_BLOCKS * 2), FOURTH_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 2), FOURTH_ZONE_Y - BLOCK_OFFSET);
    painter->drawLine(FOURTH_ZONE_X, FOURTH_ZONE_Y - BLOCK_OFFSET, FIFTH_ZONE_X + BLOCK_OFFSET, FIFTH_ZONE_Y + (NUMBER_PEGS_PER_BLOCK * NUMBER_VERTICAL_BLOCKS * 2));
    painter->setPen(QPen(black));

    //Draw the numbers and labels
    painter->setFont(QFont("courier", 4));
    painter->setBrush(QBrush(black));
    for (int i=0; i < NUMBER_VERTICAL_BLOCKS; ++i)
    {
        painter->drawText(FIRST_ZONE_X + BLOCK_OFFSET + TEXT_OFFSET, FIRST_ZONE_Y + (i * 10), QString("%1").arg(40 - ((i + 1) * NUMBER_PEGS_PER_BLOCK)));
        painter->drawText(THIRD_ZONE_X + BLOCK_OFFSET, THIRD_ZONE_Y + (i * 10) + 10, QString("%1").arg(45 + ((i + 1) * NUMBER_PEGS_PER_BLOCK)));
        painter->drawText(FIFTH_ZONE_X + BLOCK_OFFSET + TEXT_OFFSET, FIFTH_ZONE_Y + (i * 10), QString("%1").arg(125 - ((i + 1) * NUMBER_PEGS_PER_BLOCK)));
    }
    painter->drawText(SECOND_ZONE_X + NUMBER_PEGS_PER_BLOCK + TEXT_OFFSET, SECOND_ZONE_Y + (BLOCK_OFFSET * 3), QString("%1").arg(40));
    painter->drawText(SECOND_ZONE_X + (NUMBER_PEGS_PER_BLOCK * 2) + (TEXT_OFFSET * 3), SECOND_ZONE_Y + (BLOCK_OFFSET * 3), QString("%1").arg(45));
    painter->drawText(FOURTH_ZONE_X - TEXT_OFFSET, FOURTH_ZONE_Y - BLOCK_OFFSET, QString("%1").arg(85));
    painter->drawEllipse(SIXTH_ZONE_X, SIXTH_ZONE_Y, 2, 2); // The finish CribbagePeg.
    painter->drawText(SIXTH_ZONE_X - (TEXT_OFFSET * 2), SIXTH_ZONE_Y - TEXT_OFFSET, "Finish");
    painter->drawText(STAKE_ZONE_X - (TEXT_OFFSET * 7), STAKE_ZONE_Y + TEXT_OFFSET + 1, "Stakes");
    painter->drawText(SECOND_ZONE_X - TEXT_OFFSET, SECOND_ZONE_Y - (TEXT_OFFSET * 4), "Cribbage");
    painter->drawText(SECOND_ZONE_X, SECOND_ZONE_Y - (TEXT_OFFSET * 2), "Board");

    // Draw each players pegs
    for (int i=0; i < NUMBER_OF_PEGS; ++i)
    {
        m_redPegs[i]->draw(painter);
        m_bluePegs[i]->draw(painter);
    }
    for (int i=0; i < static_cast<int>(m_legendPegs.size()); ++i)
    {
        m_legendPegs[i].first->draw(painter);
        painter->drawText(m_legendPegs[i].first->x() + 2, m_legendPegs[i].first->y() + 1, m_legendPegs[i].second);
    }
}

void CribbageBoard::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect = event->rect();

    QSize newSize = rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    int side = QMIN(width(), height());
    painter.setViewport((width() - side) / 2 - event->rect().x(), (height() - side) / 2 - event->rect().y(), side, side);
    painter.setWindow(0, 0, 50, 80);
    draw(&painter);
    bitBlt(this, event->rect().topLeft(), &pixmap);
}
