/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "heartsrules.h"
#include "card.h"
#include "cardsequence.h"
#include "playerlist.h"

HeartsRules::HeartsRules(): RuleBase()
{}

HeartsRules::~HeartsRules()
{}

bool HeartsRules::isGameOver(const PlayerList& players) const
{
    for (int i=0; i < players.size(); ++i)
        if (players[i].score() >= HeartsRules::WINNING_GAME_SCORE)
            return true;
    return false;
}

bool HeartsRules::isLegalPlay(const CardSequence& sequence, const Card& cardPlayed, const Player& player) const
{
    Card::suit_t suit=Card::SUIT_ERR;

    if (player.hand().isEmpty()) // If a player doesn't have any cards, they can't really play.
        return false;
    if (sequence.isEmpty())
    {
        if (cardPlayed.isEmpty()) // Nothing has been played
            return true;
        if (player.hand().size() == HeartsRules::NUMBER_OF_CARDS_TO_DEAL)
            if ((cardPlayed.rank() == Card::TWO) && (cardPlayed.suit() == Card::CLUBS)) // First play must be 2C
                return true;
            else
                return false;
        else
            return true; // If it's not the start of the game you can lead with anything.
    }
    suit = sequence.front().suit();
    if (suit == cardPlayed.suit()) // Played a card of the same suit.
        return true;
    if (suit != cardPlayed.suit()) // Played a different suit
    {
        // Check to be sure that the don't have any card of that suit.
        int size = player.hand().size();
        for (int i=0; i < size; ++i)
            if (player.hand()[i].suit() == suit)
                return false;
    }
    return true; // So the player didn't have any cards of the proper suit.
}

bool HeartsRules::isRoundOver(const PlayerList& players) const
{
    for (int i=0; i < players.size(); ++i)
        if (! players[i].hand().isEmpty())
            return false;
    return true;
}

bool HeartsRules::isPhaseOver(const PlayerList &, const CardSequence &playSequence) const
{
    if (playSequence.size() == PHASE_OVER)
        return true;
    else
        return false;
}

int HeartsRules::cardValue(const Card& card) const
{
    if (card.suit() == Card::HEARTS)
        return 1;
    if ((card.suit() == Card::SPADES) && (card.rank() == Card::QUEEN))
        return 13;
    return 0;
}

int HeartsRules::maximumNumberOfPlayers() const
{
    return MAXIMUM_NUMBER_OF_PLAYERS;
}

int HeartsRules::minimumNumberOfPlayers() const
{
    return MINIMUM_NUMBER_OF_PLAYERS;
}

int HeartsRules::numberOfCardsToDeal() const
{
    return NUMBER_OF_CARDS_TO_DEAL;
}

int HeartsRules::winningGameScore() const
{
    return WINNING_GAME_SCORE;
}

CardSequence HeartsRules::cardPointers(const CardSequence &playSequence) const
{
    CardSequence pointers;

    // Find all the hearts and the QS.
    for (int i=0; i < playSequence.size(); ++i)
        if (playSequence[i].suit() == Card::HEARTS)
            pointers.addCard(playSequence[i]);
        else if ((playSequence[i].suit() == Card::SPADES) && (playSequence[i].rank() == Card::QUEEN))
            pointers.addCard(playSequence[i]);
    return pointers;
}

int HeartsRules::rankValue(const Card& card) const
{
    int value = RANK_ERR;

    switch (card.rank())
    {
    case Card::ACE:
        value=ACE;
        break;
    case Card::TWO:
        value=TWO;
        break;
    case Card::THREE:
        value=THREE;
        break;
    case Card::FOUR:
        value=FOUR;
        break;
    case Card::FIVE:
        value=FIVE;
        break;
    case Card::SIX:
        value=SIX;
        break;
    case Card::SEVEN:
        value=SEVEN;
        break;
    case Card::EIGHT:
        value=EIGHT;
        break;
    case Card::NINE:
        value=NINE;
        break;
    case Card::TEN:
        value=TEN;
        break;
    case Card::JACK:
        value=JACK;
        break;
    case Card::QUEEN:
        value=QUEEN;
        break;
    case Card::KING:
        value=KING;
        break;
    default:
        value=RANK_ERR;
    }
    return value;
}
