/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef GENERALGAMEAI_H
#define GENERALGAMEAI_H

#include "cardsequence.h"
#include "rulebase.h"

/**
 * This is our general play strategy when we don't know how to play the game.
 *
 * @author John Schneiderman
 */
class GeneralGameAI
{
public:
    /**
     * This constructor is used for when the play sequence does not matter in determining what card to play.
     * @param rules are the rules for the game we're playing.
     * @param hand is the hand we have.
     */
    GeneralGameAI(const RuleBase &rules, const CardSequence &hand);
    /**
     * This constructor is used for when the play sequence does matter in determining what card to play.
     * @param playSequence is the current play sequence for the game.
     * @param rules are the rules for the game we're playing.
     * @param hand is the cards we have in our hand.
     */
    GeneralGameAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand);
    /**
     * The default destructor.
     */
    ~GeneralGameAI();
    /**
     * This selects a specific number of cards from our hand based on what is a legal play.
     * @param numberOfCards is the number of cards we have to select.
     * @return the cards we have chosen.
     * @throw KardsGTError if we cannot select as many cards specified in numberOfCards.
     */
    CardSequence selectCards(int numberOfCards) const;
    /**
     * Selects cards randomly that are legal to play for the game.
     * @param nummberOfCards is the number of cards to select.
     * @note If there is no legal cards in the hand an infinite loop is created!
     * @return the cards selected.
     */
    CardSequence randomCards(int nummberOfCards) const;
    /**
     * Selects cards to play for the game.
     * @param numberOfCards is the number of cards to select.
     * @return the cards selected.
     */
    CardSequence selectCardsWithNoLegalChecks(int numberOfCards) const;
    /**
     * Selects cards randomly to play for the game without regards for the rules of the game.
     * @param nummberOfCards is the number of cards to select.
     * @return the cards selected.
     */
    CardSequence randomCardsWithNoLegalChecks(int nummberOfCards) const;

private:
    /// @param m_playSequence is the play sequence we're working with.
    CardSequence m_playSequence;
    /// @param m_hand is the hand of cards we're working with.
    CardSequence m_hand;
    /// @param m_pRules are the rules for the game.
    const RuleBase *m_pRules;
};
#endif
