/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kard.h"
#include "kardsgterror.h"
#include "karddrag.h"

#include <qpainter.h>
#include <qimage.h>
#include <qevent.h>
#include <qrect.h>

Kard::Kard(QWidget *parent, const char *name): QWidget(parent, name, WPaintClever | WRepaintNoErase), m_cardFront(), m_card()
{
    m_faceUp=m_selected=false;
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_cardBack = QImage::fromMimeSource("cardback.png");
    m_cardPath = "";
}

Kard::Kard(const Card &card, bool faceUp, QWidget *parent, const char *name): QWidget(parent, name, WPaintClever | WRepaintNoErase), m_cardFront()
{
    m_selected=false;
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_cardBack=QImage::fromMimeSource("cardback.png");
    setCard(card);
    setFaceUp(faceUp);
    m_cardPath = "";
}

Kard::~Kard()
{}

QSize Kard::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const QPixmap& Kard::cardBack() const
{
    return m_cardBack;
}

const QPixmap& Kard::cardFront() const
{
    return m_cardFront;
}

bool Kard::isFaceUp() const
{
    return m_faceUp;
}

bool Kard::isSelected() const
{
    return m_selected;
}

void Kard::setCardBack(const QPixmap &cardBack)
{
    if (cardBack.isNull())
        throw KardsGTError("Kard", "setCardBack", "cardBack is null!");
    else
        m_cardBack=cardBack;
    update();
    updateGeometry();
}

void Kard::setCardBack(const QString &cardBackFilename)
{
    // If we pass in a null string use the default
    if (cardBackFilename.isEmpty())
        m_cardBack = QImage::fromMimeSource("cardback.png");
    else
        m_cardBack = QImage::fromMimeSource(cardBackFilename);
    if (m_cardBack.isNull())
    {
        m_cardBack = QImage(cardBackFilename);
        if (cardBackFilename.isEmpty())
            throw KardsGTError(QString("Kard"), QString("setCardBack"), QString("cardBackFilename %1 was not found!").arg(cardBackFilename));
    }
    update();
    updateGeometry();
}

void Kard::setCardFront(const QPixmap &cardFront)
{
    if (cardFront.isNull())
        throw KardsGTError("Kard", "setCardFront", "cardFront is null!");
    else
        m_cardFront=cardFront;
    update();
    updateGeometry();
}

void Kard::setCardPath(const QString &cardPath)
{
    m_cardPath = cardPath;
    setCard(m_card);
}

void Kard::setCard(const Card &card)
{
    m_card = card;
    if (! card.isEmpty())
    {
        if ((m_cardPath == "default") || m_cardPath.isEmpty())
            m_cardFront = QImage::fromMimeSource(card.toString().lower() + ".xpm");
        else
            m_cardFront = QImage(m_cardPath + card.toString().lower() + ".xpm");
        if (m_cardFront.isNull())
            throw KardsGTError(QString("Kard"), QString("setCard"), QString("%1.xpm image not found!").arg(card.toString().lower()));
    }
    else
        m_cardFront.fill();
    update();
    updateGeometry();
}

void Kard::setFaceUp(bool faceUp)
{
    m_faceUp=faceUp;
    update();
    updateGeometry();
}

void Kard::setSelected(bool select)
{
    // Make sure we only emit the signal once.
    if (m_selected == select)
        return;

    m_selected=select;
    if (m_selected)
        emit cardSelected(*this);
    update();
    updateGeometry();
}

void Kard::flipCard()
{
    if (m_faceUp)
        m_faceUp=false;
    else
        m_faceUp=true;
    update();
    updateGeometry();
}

void Kard::draw(QPainter &painter)
{
    //Draw a plane rectangle if nothing is set, else draw either the front of the card or the back of the card.
    if (m_card.isEmpty())
        painter.drawRect(this->rect());
    else if (m_faceUp)
        painter.drawPixmap(QRect(0, 0, PREFERRED_WIDTH, PREFERRED_HEIGHT), m_cardFront);
    else
        painter.drawPixmap(QRect(0, 0, PREFERRED_WIDTH, PREFERRED_HEIGHT), m_cardBack);
}

Kard& Kard::operator=(const Kard &kard)
{
    m_cardBack=kard.m_cardBack;
    m_cardFront=kard.m_cardFront;
    m_faceUp=kard.m_faceUp;
    m_selected=kard.m_selected;
    m_card=kard.m_card;
    update();
    updateGeometry();
    return *this;
}

void Kard::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect=event->rect();

    QSize newSize=rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    painter.translate(-rect.x(), -rect.y());
    draw(painter);
    bitBlt(this, rect.x(), rect.y(), &pixmap, 0, 0, rect.width(), rect.height());
}

void Kard::mouseDoubleClickEvent(QMouseEvent *event)
{
    // Prevent a card from being selected more than once.
    if (m_selected)
        return;
    // User selected a card.
    if (event->button() == QMouseEvent::LeftButton)
        setSelected(true);
}

void Kard::mousePressEvent(QMouseEvent *event)
{
    // Drag our kard.
    if (event->button() == LeftButton)
        m_dragPosition = event->pos();
    else
        QWidget::mousePressEvent(event);
}

void Kard::mouseMoveEvent(QMouseEvent *event)
{
    // Are we dragging our kard with the left mouse button?
    if (event->state() & LeftButton)
    {
        int distance = (event->pos() - m_dragPosition).manhattanLength();

        if (distance > START_DRAG_DISTANCE)
            startDrag();
    }
}

void Kard::dragEnterEvent(QDragEnterEvent *event)
{
    event->accept(event->provides("application/x-kard")); // This is the mime type used to identify our Kard object.
}

void Kard::startDrag()
{
    KardDrag *drag = new KardDrag(*this, this->parentWidget(), m_card.toString());
    drag->drag();
}

const Card& Kard::card()
{
    return m_card;
}
