/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardpile.h"
#include "kard.h"
#include "cardpile.h"
#include "karddrag.h"
#include "kardsgterror.h"
#include "kardsequence.h"

#include <qpainter.h>
#include <qevent.h>
#include <qlayout.h>

KardPile::KardPile(QWidget *parent, const char *name): QWidget(parent, name, WPaintClever | WRepaintNoErase), m_kards()
{
    setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    m_pCardPile=new CardPile();
    m_topCardUp=false;
    m_cardImagePath = "";
    m_cardBackImage = "";
    setAcceptDrops(true);
    m_pLayout=new QVBoxLayout(this, MARGIN, HORIZONTAL, "KardSequenceLayout");
}

KardPile::~KardPile()
{
    int size=m_kards.size();

    if (m_pCardPile != NULL)
        delete m_pCardPile;
    for (int i=0; i < size; ++i)
        delete m_kards[i];
    delete m_pLayout;
}

QSize KardPile::sizeHint() const
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

bool KardPile::isTopCardUp() const
{
    return m_topCardUp;
}

void KardPile::setTopCardUp(bool topUp)
{
    m_topCardUp=topUp;
    update();
    updateGeometry();
}

const CardPile&  KardPile::cardPile() const
{
    return *m_pCardPile;
}

void KardPile::clear()
{
    int size=m_kards.size();

    for (int i=0; i < size; ++i)
        removeKard(*m_kards[0]);
    m_kards.clear();
    if (m_pCardPile != NULL)
        m_pCardPile->clear();
    delete m_pLayout;
    m_pLayout=0;
    m_pLayout=new QVBoxLayout(this, MARGIN, HORIZONTAL, "KardSequenceLayout");
    update();
    updateGeometry();
}

void KardPile::selectTopCard()
{
    m_kards.front()->setSelected(true);
}

void KardPile::setCardPile(const CardPile &cardPile)
{
    // Don't bother if we're already the same.
    if ((*m_pCardPile) == cardPile)
        return;

    CardPile pile=cardPile;
    clear();
    while (! pile.isEmpty())
        addCardToBottom(pile.removeCard()); // Add to the bottom so that the top card in cardPile will still be the top card.
    update();
    updateGeometry();
}

void KardPile::addCardToBottom(const Card &card)
{
    Kard *kard = new Kard(card, false, this, card.toString());

    m_pCardPile->addCardToBottom(card);
    m_pLayout->addWidget(kard);
    kard->setCardBack(m_cardBackImage);
    kard->setCardPath(m_cardImagePath);
    m_kards.push_back(kard);
    update();
    updateGeometry();
    connect(kard, SIGNAL(cardSelected(Kard &)), this, SIGNAL(kardSelected(Kard &)));
}

void KardPile::paintEvent(QPaintEvent *event)
{
    static QPixmap pixmap;
    QRect rect=event->rect();

    QSize newSize=rect.size().expandedTo(pixmap.size());
    pixmap.resize(newSize);
    pixmap.fill(this, rect.topLeft());

    QPainter painter(&pixmap, this);
    painter.translate(-rect.x(), -rect.y());

    if ((m_pCardPile == NULL) || (m_pCardPile->size() == 0))
        painter.drawRect(rect);
    else
        draw(painter);
    bitBlt(this, rect.x(), rect.y(), &pixmap, 0, 0, rect.width(), rect.height());
}

void KardPile::draw(QPainter &painter)
{
    int size=m_kards.size();

    painter.drawRect(painter.window());
    if (size > 0)
    {
        m_kards.front()->setFaceUp(m_topCardUp);
        m_kards.front()->show();
        if (size > 1)
            painter.drawPixmap(m_kards.front()->pos().x() + 1, m_kards.front()->pos().y() + 1, QImage::fromMimeSource("cardback.png"));
    }
}

void KardPile::addKard(Kard &kard)
{
    addCardToBottom(kard.card());
}

void KardPile::removeKard(Kard &kard)
{
    if (m_pCardPile->topCard() == kard.card())
    {
        removeCard();
        update();
        updateGeometry();
    }
    else
        throw KardsGTError("KardPile", "removeKard", "kard passed in was not the top card.");
}

void KardPile::removeCard()
{
    vector<Kard *>::iterator iter=m_kards.begin();

    (*iter)->hide();
    m_pLayout->remove(*iter);
    m_pCardPile->removeCard();
    m_kards.erase(iter);
}

void KardPile::dragEnterEvent(QDragEnterEvent *event)
{
    event->accept(event->provides("application/x-kard"));
}

void KardPile::setSelected(const Card &card)
{
    if (m_pCardPile->topCard() == card) // In a pile of cards only the top card is allowed to be selected.
        m_kards[0]->setSelected(true);
    else
        throw KardsGTError("KardPile", "setSelected", "card was not the top card selected!");
}

void KardPile::dropEvent(QDropEvent *event)
{
    Kard kard;

    if (KardDrag::decode(event, kard))
    {
        QWidget *fromWidget = event->source();

        if (fromWidget && (fromWidget != this) && fromWidget->inherits("KardPile"))
        {
            KardPile *fromKardPile = static_cast<KardPile *>(fromWidget);
            fromKardPile->setSelected(kard.card());
        }
        else if (fromWidget && (fromWidget != this) && fromWidget->inherits("KardSequence"))
        {
            KardSequence *fromKardSequence = static_cast<KardSequence *>(fromWidget);
            fromKardSequence->setSelected(kard.card());
        }
    }
}

void KardPile::setBackImage(const QString &backImage)
{
    m_cardBackImage = backImage;
}

void KardPile::setCardImagePath(const QString &imagePath)
{
    m_cardImagePath = imagePath;
}
