/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "abigail.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
// Games Abigail knows how to play
#include "abigailcribbageai.h"
#include "abigailheartsai.h"
#include "abigailspadesai.h"

Abigail::Abigail(): AIBase()
{}

Abigail::~Abigail()
{}

CardSequence Abigail::whichCards(const QString& game, const CardSequence& playSequence, const RuleBase& rules, const CardSequence& hand, const GameInformation &specialInformation) const
{
    CardSequence cards;

    if (game.contains("cribbage"))
        cards = handleCribbageCards(game, playSequence, rules, hand);
    else if (game.contains("spades"))
        cards = handleSpadesCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("hearts"))
        cards = handleHeartsCards(game, playSequence, rules, hand);
    else
    {
        BasicGameStrategies ai(rules);
        cards=ai.selectCards(1, hand, playSequence);
    }
    return cards;
}

int Abigail::bidAt(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    int bid = -1;

    if (game.contains("spades"))
        bid = handleSpadesBid(game, playSequence, rules, hand, specialInformation, score);
    else
    {
        BasicGameStrategies ai(rules);
        bid=ai.randomBid(1, 100);
    }
    return bid;
}

Player::skill_Level Abigail::skillLevel(const QString& game) const
{
    if (game == "spades")
        return Player::Professional;
    else if (game == "cribbage")
        return Player::Apprentice;
    else if (game == "hearts")
        return Player::Novice;
    else
        return Player::Newb;
}

QString Abigail::name() const
{
    return "Abigail";
}

CardSequence Abigail::handleCribbageCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "cribbage")
    {
        AbigailCribbageAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "cribbageDealerCrib")
    {
        AbigailCribbageAI ai(playSequence, rules, hand);
        return ai.myCrib();
    }
    else if (game == "cribbageNonDealerCrib")
    {
        AbigailCribbageAI ai(playSequence, rules, hand);
        return ai.opponentsCrib();
    }
    throw KardsGTError("Abigail", "handleCribbageCards", "Failed to determine game type or phase.");
}

CardSequence Abigail::handleSpadesCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "spades")
    {
        AbigailSpadesAI ai(playSequence, rules, hand);
        // If we bid nil play differently
        if (specialInformation.value("Made Nil Bid") == "true")
            return ai.selectCardsForNilBid();
        else
            return ai.selectCards();
    }
    else if (game == "spadesExchangeCards")
    {
        AbigailSpadesAI ai(playSequence, rules, hand);
        return ai.exchangeCards();
    }
    throw KardsGTError("Abigail", "handleSpadesCards", "Failed to determine game type or phase.");
}

CardSequence Abigail::handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "hearts")
    {
        AbigailHeartsAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "heartsPassPhase")
    {
        AbigailHeartsAI ai(playSequence, rules, hand);
        return ai.passCards();
    }
    throw KardsGTError("Abigail", "handleHeartsCards", "Failed to determine game type or phase.");
}

int Abigail::handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "spadesBidDoubleNil")
    {
        AbigailSpadesAI ai(playSequence, rules, hand);
        return ai.bidDoubleNil(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "spadesBidNil")
    {
        AbigailSpadesAI ai(playSequence, rules, hand);
        return ai.bidNil(specialInformation.value("Partners Bid").toInt());
    }
    throw KardsGTError("Abigail", "handleSpadesBid", "Failed to determine game type or phase.");
}
