/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "playerlist.h"
#include "kardsgterror.h"
#include "player.h"

#include <iterator>
#include <qtextstream.h>

PlayerList::PlayerList(): m_playerList()
{
    m_current=-1;
}

void PlayerList::insert(Player &player)
{
    m_playerList.push_back(&player);
    m_current=0;
}

bool PlayerList::remove
    (const Player &player)
{
    vector<Player *>::iterator iter=m_playerList.begin();

    while (iter != m_playerList.end())
        if (*(*iter) == player)
        {
            m_playerList.erase(iter);
            return true;
        }
        else
            iter++;
    return false;
}

Player& PlayerList::nextPlayer()
{
    m_current=(m_current + 1) % m_playerList.size();
    return currentPlayer();
}

Player& PlayerList::nextNonEmptyPlayer()
{
    bool playersHaveCards=false;
    int size=m_playerList.size();

    // See if at least one player still has cards.
    for (int i=0; i < size; i++)
        if (!m_playerList[i]->hand().isEmpty())
            playersHaveCards=true;

    // Find the next player with cards
    if (playersHaveCards)
    {
        m_current=(m_current + 1) % m_playerList.size();
        while (m_playerList[m_current]->hand().isEmpty())
            m_current=(m_current + 1) % m_playerList.size();
    }
    else
        throw KardsGTError("PlayerList", "nextPlayer", "Could not find any players with cards!");
    return currentPlayer();
}

Player& PlayerList::currentPlayer()
{
    return *m_playerList[m_current];
}

Player& PlayerList::rightPlayer()
{
    int player=(m_current - 1) % m_playerList.size();

    return *m_playerList[player];
}

Player& PlayerList::leftPlayer()
{
    int player=(m_current + 1) % m_playerList.size();

    return *m_playerList[player];
}

Player& PlayerList::rightPlayerWithCards()
{
    int playerIndex=(m_current - 1) % m_playerList.size();

	while (m_playerList[playerIndex]->hand().isEmpty())
		playerIndex=(m_current - 1) % m_playerList.size();
	if (playerIndex == m_current)
        throw KardsGTError("PlayerList", "rightPlayerWithCards", "Could not find any players with cards!");
    return *m_playerList[playerIndex];
}

Player& PlayerList::leftPlayerWithCards()
{
    int playerIndex=(m_current + 1) % m_playerList.size();

	while (m_playerList[playerIndex]->hand().isEmpty())
		playerIndex = (playerIndex + 1) % m_playerList.size();
	if (playerIndex == m_current)
        throw KardsGTError("PlayerList", "leftPlayerWithCards", "Could not find any players with cards!");
    return *m_playerList[playerIndex];
}

void PlayerList::setCurrentPlayer(const Player &player)
{
    int size=m_playerList.size();

    // Search for the player and set him as the current player.
    for (int i=0; i < size; i++)
        if ((*m_playerList[i]) == player)
        {
            m_current=i;
            return;
        }
    throw KardsGTError(QString("PlayerList"), QString("setCurrentPlayer"), QString("%1 was not found in player list.").arg(player.name()));
}

int PlayerList::size() const
{
    return m_playerList.size();
}

Player& PlayerList::operator[](int index)
{
    if ((index < 0) || (index >= static_cast<int>(m_playerList.size())))
        throw KardsGTError(QString("PlayerList"), QString("operator[]"), QString("An index of %1 is out of bounds.").arg(index));
    return *m_playerList[index];
}

const Player& PlayerList::operator[](int index) const
{
    if ((index < 0) || (index >= static_cast<int>(m_playerList.size())))
        throw KardsGTError(QString("PlayerList"), QString("const operator[]"), QString("An index of %1 is out of bounds.").arg(index));
    return *m_playerList[index];
}

void PlayerList::clear()
{
    m_current=-1;
    m_playerList.clear();
}

QTextStream& operator<<(QTextStream &out, const PlayerList &playerList)
{
    int size=playerList.size();

    out << "NumberOfPlayers: " << size << endl;
    for (int i=0; i < (size - 1); ++i)
        out << playerList[i] << endl;
    out << playerList[size - 1] << endl; // last player in our list.
    out << "CurrentPlayer: "<< playerList.m_current;
    return out;
}

ostream& operator<<(ostream &out, const PlayerList &playerList)
{
    int size=playerList.size();

    out << "NumberOfPlayers: " << size << endl;
    for (int i=0; i < (size - 1); ++i)
        out << playerList[i] << endl;
    out << playerList[size - 1] << endl; // last player in our list.
    out << "CurrentPlayer: "<< playerList.m_current;
    return out;
}

QTextStream& operator>>(QTextStream &in, PlayerList &playerList)
{
    int size=0;
    QString label;

    in >> label >> size;
    for (int i=0; i < size; ++i)
        in >> playerList[i];
    in >> label >> playerList.m_current;
    return in;
}

bool PlayerList::isEmpty() const
{
    if (m_playerList.size() == 0)
        return true;
    return false;
}

void PlayerList::swap(int left, int right)
{
    if (left == right)
        return;

    Player *temp=m_playerList[right];

    m_playerList[right]=m_playerList[left];
    m_playerList[left]=temp;
}
