/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "jack.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
// Games Jack knows how to play
#include "jackcribbageai.h"
#include "jackheartsai.h"
#include "jackspadesai.h"

Jack::Jack(): AIBase()
{}

Jack::~Jack()
{}

CardSequence Jack::whichCards(const QString& game, const CardSequence& playSequence, const RuleBase& rules, const CardSequence& hand, const GameInformation &specialInformation) const
{
    CardSequence cards;

    if (game.contains("cribbage"))
        cards = handleCribbageCards(game, playSequence, rules, hand);
    else if (game.contains("spades"))
        cards = handleSpadesCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("hearts"))
        cards = handleHeartsCards(game, playSequence, rules, hand);
    else
    {
        BasicGameStrategies ai(rules);
        cards=ai.selectCards(1, hand, playSequence);
    }
    return cards;
}

int Jack::bidAt(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    int bid = -1;

    if (game.contains("spades"))
        bid = handleSpadesBid(game, playSequence, rules, hand, specialInformation, score);
    else
    {
        BasicGameStrategies ai(rules);
        bid=ai.randomBid(1, 100);
    }
    return bid;
}

Player::skill_Level Jack::skillLevel(const QString& game) const
{
    if (game == "spades")
        return Player::Novice;
    else if (game == "cribbage")
        return Player::Novice;
    else if (game == "hearts")
        return Player::Amateur;
    else if (game == "old maid")
        return Player::Master;
    else
        return Player::Newb;
}

QString Jack::name() const
{
    return "Jack";
}

CardSequence Jack::handleCribbageCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "cribbage")
    {
        JackCribbageAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "cribbageDealerCrib")
    {
        JackCribbageAI ai(playSequence, rules, hand);
        return ai.myCrib();
    }
    else if (game == "cribbageNonDealerCrib")
    {
        JackCribbageAI ai(playSequence, rules, hand);
        return ai.opponentsCrib();
    }
    throw KardsGTError("Jack", "handleCribbageCards", "Failed to determine game type or phase.");
}

CardSequence Jack::handleSpadesCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "spades")
    {
        JackSpadesAI ai(playSequence, rules, hand);
        // If we bid nil play differently
        if (specialInformation.value("Made Nil Bid") == "true")
            return ai.selectCardsForNilBid();
        else
            return ai.selectCards();
    }
    else if (game == "spadesExchangeCards")
    {
        JackSpadesAI ai(playSequence, rules, hand);
        return ai.exchangeCards();
    }
    throw KardsGTError("Jack", "handleSpadesCards", "Failed to determine game type or phase.");
}

CardSequence Jack::handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "hearts")
    {
        JackHeartsAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "heartsPassPhase")
    {
        JackHeartsAI ai(playSequence, rules, hand);
        return ai.passCards();
    }
    throw KardsGTError("Jack", "handleHeartsCards", "Failed to determine game type or phase.");
}

int Jack::handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "spadesBidDoubleNil")
    {
        JackSpadesAI ai(playSequence, rules, hand);
        return ai.bidDoubleNil(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "spadesBidNil")
    {
        JackSpadesAI ai(playSequence, rules, hand);
        return ai.bidNil(specialInformation.value("Partners Bid").toInt());
    }
    throw KardsGTError("Jack", "handleSpadesBid", "Failed to determine game type or phase.");
}
