/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CRIBBAGEINTERFACE_H
#define CRIBBAGEINTERFACE_H

#include "cribbageinterfacebase.h"
#include "cribbage.h"

#include <vector>
using std::vector;
using std::pair;

class Card;
class Kard;
class Player;
class QString;
class CardSequence;

/**
 * This is our Cribbage game interface.
 *
 * @author John Schneiderman
 */
class CribbageInterface: public CribbageInterfaceBase, public Cribbage
{
    Q_OBJECT

public:
    /**
     * This is the default destructor.
     * @param profileDatabase is the user profile database.
     * @param parent is our parent widget.
     * @param name is the name of our widget.
     */
    CribbageInterface(UserProfileDatabase &profileDatabase, QWidget* parent = 0, const char* name = 0);
    /**
     * The destructor.
     */
    ~CribbageInterface();
    /**
     * This updates all the values displayed on our game table.
     */
    void updateTable();
    /**
     * This request cards from a player.
     * @param prompt is the prompt for the player.
     * @param player is the player to request the cards from.
     * @param numberOfCards is the number of cards we want.
     * @return the cards the player has selected.
     */
    CardSequence requestCards(const QString &prompt, const Player &player, int numberOfCards);
    /**
     * This displays a message to a player.
     * @param player is the player to see the message.
     * @param message is the message for the player to see.
     */
    void displayMessage(const Player &player, const QString &message);
    /**
     * This prompts a message.
     * @param caption is caption for the message displayed.
     * @param message is the message for the player to see.
     */
    void promptMessage(const QString &caption, const QString &message);
    /**
     * This displays the round summary.
     * @param caption is the caption for the display window.
     * @param displayHands are the hands to display.
     * @param messages are the messages to accompany each hand.
     */
    void displayRoundSummary(const QString &caption, const vector<CardSequence> &displayHands, const vector<QString> &messages);
    /**
     * This resets our game table.
     */
    void resetGame();
    /**
     * This will update the stakes of the cribbage board, and updates the player skill ratings.
     * @note We're giving players credit for winning a stake to give a better reflection of their skills in Cribbage.
     * @param roundWinnerIndex is the index of the winner of the round, if it is -1 then the player database is not changed.
     */
    void updateStakes(int roundWinnerIndex);
    /**
     * Displays a message for the player telling them if they won or not, and updates the player skill ratings.
     * @throw KardsGTError if we cannot determine who won the game.
     */
    void gameWon();
    /**
     * Gives the previous scores for the two pegs.
     * @return the first int is the red pegs previous score, and the second int is the blue pegs previous score.
     */
    pair<int, int> previousScores() const;
    /**
     * Sets the previous scores for the two pegs.
     * @param redScore is the previous score for the red peg.
     * @param blueScore is the previous score for the blue peg.
     */
    void setPreviousScores(int redScore, int blueScore);
    /**
     * Shows or hides the starter card.
     * @param show is true if you want the starter car to be shown, false elsewise.
     */
    void showStarterCard(bool show);

protected:
    /**
     * This currently handles the computer playing mechanism.
     */
    void computerPlay();

protected slots:
    /**
     * This slot handles the kard selected by player 1.
     * @param kard is the card the player selected.
     */
    void player1CardPlayed(Kard &kard);
    /**
     * This slot handles the kard selected by player 2.
     * @param kard is the card the player selected.
     */
    void player2CardPlayed(Kard &kard);
    /**
     * Saves the player 1's new card arrangement.
     */
    void player1CardMoved();
    /**
     * Saves the player 2's new card arrangement.
     */
    void player2CardMoved();

private:
    /**
     * These are the number of ms to launch each timer.
     * @param COMPUTER_PLAYER_TIME is how often to check to see if a computer player can play.
     * @param CLEARING_DELAY_TIME is how long our delay should be before we clear the playing sequence. @note This should be less than the time for a computer player to play.
     */
    enum TimerLengths { COMPUTER_PLAYER_TIME=1500, CLEARING_DELAY_TIME=500 };

    /// @param m_computerTimerId is the id for the timer for the computer player.
    int m_computerTimerId;
    /// @param m_clearingDelayId is the id for the timer for the clearing delay.
    int m_clearingDelayId;
    /// @param m_pUserProfile is the user profile database.
    const UserProfileDatabase *m_pUserProfile;

    /**
     * This handles the timer events.
     * @param event is the generating event.
     */
    void timerEvent(QTimerEvent *event);
};
#endif
