/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "euchrerules.h"
#include "playerlist.h"
#include "kardsgterror.h"

EuchreRules::EuchreRules()
{
    m_leftBowerSuit = m_trumpSuit = Card::SUIT_ERR;
    m_alone = false;
}

EuchreRules::~EuchreRules()
{}

bool EuchreRules::isGameOver(const PlayerList& players) const
{
    for (int index = 0, size = players.size(); index < size; ++index)
        if (players[index].score() == POINTS_TO_WIN)
            return true;
    return false;
}

bool EuchreRules::isLegalPlay(const CardSequence& sequence, const Card& cardPlayed, const Player& player) const
{
    // Any card can be played on the initial play
    if (sequence.isEmpty())
        return true;
    Card::Suit leadSuit = sequence.front().suit();
    Card leftBower = Card(Card::JACK, m_leftBowerSuit);

    // Must play suit if you can
    if ((cardPlayed.suit() == leadSuit) || ((leadSuit == m_trumpSuit) && (cardPlayed == leftBower)))
        return true;
    else // If you don't have the lead suit or the left bower when trump is lead, then you can play anything
        if (! player.hand().hasSuit(leadSuit))
            if (leadSuit == m_trumpSuit) // If trump is lead we have to check for the left bower.
                if (player.hand().hasCard(leftBower)) // If the player has the left bower he has to play because trump was lead.
                    return false;
                else
                    return true;
            else
                return true;
    return false;
}

bool EuchreRules::isPhaseOver(const PlayerList &, const CardSequence &playSequence) const
{
    if (playSequence.size() == PHASE_OVER)
        return true;
    else if (m_alone && (playSequence.size() == PHASE_OVER_LONE_MAKER))
        return true;
    return false;
}

bool EuchreRules::isRoundOver(const PlayerList& players) const
{
    for (int index = 0, size = players.size(); index < size; ++index)
        if (! players[index].hand().isEmpty())
            return false;
    return true;
}

int EuchreRules::cardValue(const Card &) const
{
    throw KardsGTError("EuchreRules", "cardValue", "This is not implemented!");
    return -1;
}

int EuchreRules::rankValue(const Card& card) const
{
    int value = -1;

    // Trump is (Spades): JS JC A K Q 10 9 8 7
    // Non-trump is: A K Q J(not left bower) 10 9 8 7
    if ((card.suit() == m_trumpSuit) || ((card.rank() == Card::JACK) && (card.suit() == m_leftBowerSuit)))
    {
        if ((card.rank() == Card::JACK) && (card.suit() == m_trumpSuit)) // Right Bower
            value = 17;
        else if ((card.rank() == Card::JACK) && (card.suit() == m_leftBowerSuit)) // Left Bower
            value = 16;
        else
            switch (card.rank())
            {
            case Card::ACE:
                value = 15;
                break;
            case Card::KING:
                value = 14;
                break;
            case Card::QUEEN:
                value = 13;
                break;
            case Card::TEN:
                value = 12;
                break;
            case Card::NINE:
                value = 11;
                break;
            case Card::EIGHT:
                value = 10;
                break;
            case Card::SEVEN:
                value = 9;
                break;
            default:
                throw KardsGTError("EuchreRules", "rankValue", "Invalid rank value!");
            }
    }
    else
    {
        switch (card.rank())
        {
        case Card::ACE:
            value = 8;
            break;
        case Card::KING:
            value = 7;
            break;
        case Card::QUEEN:
            value = 6;
            break;
        case Card::JACK:
            value = 5;
            break;
        case Card::TEN:
            value = 4;
            break;
        case Card::NINE:
            value = 3;
            break;
        case Card::EIGHT:
            value = 2;
            break;
        case Card::SEVEN:
            value = 1;
            break;
        default:
            throw KardsGTError("EuchreRules", "rankValue", "Invalid rank value!");
        }
    }
    return value;
}

int EuchreRules::maximumNumberOfPlayers() const
{
    return MAXIMUM_PLAYERS;
}

int EuchreRules::minimumNumberOfPlayers() const
{
    return MINIMUM_PLAYERS;
}

int EuchreRules::numberOfCardsToDeal(int sizeOfHand) const
{
    if (sizeOfHand < 3)
        return 3;
    else
        return 2;
}

int EuchreRules::winningGameScore() const
{
    return POINTS_TO_WIN;
}

Card::Suit EuchreRules::trumpSuit() const
{
    return m_trumpSuit;
}

Card::Suit EuchreRules::leftBowerSuit() const
{
    return m_leftBowerSuit;
}

void EuchreRules::setTrumpSuit(Card::Suit trumpSuit)
{
    m_trumpSuit = trumpSuit;
    switch (trumpSuit)
    {
    case Card::SPADES:
        m_leftBowerSuit = Card::CLUBS;
        break;
    case Card::HEARTS:
        m_leftBowerSuit = Card::DIAMONDS;
        break;
    case Card::CLUBS:
        m_leftBowerSuit = Card::SPADES;
        break;
    case Card::DIAMONDS:
        m_leftBowerSuit = Card::HEARTS;
        break;
    default:
        m_leftBowerSuit = Card::SUIT_ERR;
    }
}

int EuchreRules::pointsToWinRound() const
{
    return POINTS_TO_WIN_ROUND;
}

bool EuchreRules::isMarch(int score) const
{
    return score == MARCH;
}

int EuchreRules::makerAloneMarchPoint() const
{
    return MAKER_ALONE_MARCH;
}

int EuchreRules::makerAlonePoint() const
{
    return MAKER_ALONE;
}

int EuchreRules::makerMarchPoint() const
{
    return MAKER_MARCH;
}

int EuchreRules::makerPoint() const
{
    return MAKER;
}

int EuchreRules::defenderAloneMarchPoint() const
{
    return DEFENDER_ALONE_MARCH;
}

int EuchreRules::defenderAlonePoint() const
{
    return DEFENDER_ALONE;
}

int EuchreRules::defenderPoint() const
{
    return DEFENDER;
}

bool EuchreRules::isLeftBower(const Card &card) const
{
    return ((card.rank() == Card::JACK) && (card.suit() == m_leftBowerSuit));
}

void EuchreRules::setIsAlone(bool alone)
{
    m_alone = alone;
}
