/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "oldmaidinterface.h"
#include "oldmaidplayer.h"
#include "kardselection.h"
#include "kard.h"
#include "kardmessagedisplay.h"

#include <qmessagebox.h>

OldMaidInterface::OldMaidInterface(UserProfileDatabase &profileDatabase, QWidget *parent, const char *name): OldMaidInterfaceBase(parent, name), OldMaid(this, profileDatabase)
{
    m_pUserProfile = &profileDatabase;
    setBackgroundColor(darkGreen);
    // Setup pointers to  interface components
    m_kardPlayers.push_back(player1);
    m_kardPlayers.push_back(player2);
    m_kardPlayers.push_back(player3);
    m_kardPlayers.push_back(player4);
    m_kardPlayers.push_back(player5);
    m_kardPlayers.push_back(player6);
    m_kardPlayers.push_back(player7);
    m_kardPlayers.push_back(player8);
    m_playersHands.push_back(playerHand1);
    m_playersHands.push_back(playerHand2);
    m_playersHands.push_back(playerHand3);
    m_playersHands.push_back(playerHand4);
    m_playersHands.push_back(playerHand5);
    m_playersHands.push_back(playerHand6);
    m_playersHands.push_back(playerHand7);
    m_playersHands.push_back(playerHand8);
    // Connect the player's cards to the selected slots
    connect(playerHand1, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand2, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand3, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand4, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand5, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand6, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand7, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    connect(playerHand8, SIGNAL(kardSelected(Kard &)), this, SLOT(playerCardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(playerHand1, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand2, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand3, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand4, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand5, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand6, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand7, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    connect(playerHand8, SIGNAL(kardMoved()), this, SLOT(playerCardMoved()));
    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
}

OldMaidInterface::~OldMaidInterface()
{
    int size = m_kardPlayers.size();

    for (int index = 0; index < size; ++index)
        m_kardPlayers[index] = NULL;
    for (int index = 0; index < size; ++index)
        m_playersHands[index] = NULL;
}

void OldMaidInterface::updateTable()
{
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
    {
        m_playersHands[index]->setBackImage(m_pUserProfile->pathToCardBackImage(m_players[m_humanPlayerIndex].name()));
        m_playersHands[index]->setCardImagePath(m_pUserProfile->pathToCardImages(m_players[m_humanPlayerIndex].name()));
        m_playersHands[index]->setCardSequence(m_players[index].hand());
        if (index != m_humanPlayerIndex)
            m_kardPlayers[index]->setPlayerImage(m_players[index].name().lower() + ".png");
        else
        {
            m_kardPlayers[index]->setPlayerImage(m_pUserProfile->userImage(m_players[index].name()));
            m_playersHands[index]->setFaceUp(true);
        }
        m_kardPlayers[index]->setDealer(m_players[index].isDealer());
        m_kardPlayers[index]->setTurn(m_players[index].isTurn());
        m_kardPlayers[index]->setLevel(m_players[index].level());
    }
}

CardSequence OldMaidInterface::discardPairs()
{
    CardSequence pairs, pairsFound;
    CardSequence hand = m_players[m_humanPlayerIndex].hand();
    int size = hand.size() - 1;

    sort(hand.begin(), hand.end());
    for (int index=0; index < size; ++index)
        if (m_rules.rankValue(hand[index]) == m_rules.rankValue(hand[index + 1]))
        {
            pairsFound.addCard(hand[index]);
            pairsFound.addCard(hand[index + 1]);
            ++index;
        }
    if (! pairsFound.isEmpty())
    {
        KardSelection *selectionInterface=new KardSelection("Select All Pairs", pairsFound.size(), m_players[m_humanPlayerIndex].hand(), m_pUserProfile->pathToCardImages(m_players[m_humanPlayerIndex].name()), m_pUserProfile->pathToCardBackImage(m_players[m_humanPlayerIndex].name()), this, "discardPairs");

        if (selectionInterface->exec())
            return selectionInterface->selection();
        else
            return discardPairs();
    }
    return pairs;
}

void OldMaidInterface::displayMessage(const Player &player, const QString &message)
{
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
        if (m_players[index].name() == player.name())
        {
            m_kardPlayers[index]->setCaption(message);
            break;
        }
    updateTable();
}

void OldMaidInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
    updateTable();
}

void OldMaidInterface::gameWon()
{
    int index = 0;

    // Find the looser
    for (int size = m_players.size(); index < size; ++index)
        if (! m_players[index].hand().isEmpty())
        {
            if (index == m_humanPlayerIndex)
            {
                promptMessage("Game Over", "You are the old maid!");
                m_pProfileDatabase->setUserGamesWon(m_players[m_humanPlayerIndex].name(), "old maid", false);
            }
            else
            {
                m_playersHands[index]->setFaceUp(true);
                promptMessage("Game Over", m_players[index].name() + " is the old maid!");
                m_pProfileDatabase->setUserGamesWon(m_players[m_humanPlayerIndex].name(), "old maid", true);
            }
            break;
        }
}

void OldMaidInterface::reset()
{
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
    {
        if (index != m_humanPlayerIndex)
            m_playersHands[index]->setFaceUp(false);
        else
            m_playersHands[index]->setFaceUp(true);
        m_kardPlayers[index]->setLevel(m_players[index].level());
    }
}

void OldMaidInterface::showDiscardedPairs(const QString &name, const CardSequence &pairs)
{
    KardMessageDisplay message("Discarded Pairs", pairs, QString(name + "'s pairs."), m_pUserProfile->pathToCardImages(m_players[m_humanPlayerIndex].name()), m_pUserProfile->pathToCardBackImage(m_players[m_humanPlayerIndex].name()), this, "showDiscardedPairs");

    message.exec();
    updateTable();
}

void OldMaidInterface::playerCardPlayed(Kard &kard)
{
    int currentPlayerIndex = 0;
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
        if (m_players[index].isTurn())
        {
            currentPlayerIndex = index;
            break;
        }

    // Make sure it's their turn.
    if (! m_playersHands[leftPlayerIndex()]->cardSequence().hasCard(kard.card()))
    {
        promptMessage("Warning", m_players[currentPlayerIndex].name() + " you must select a card from the player to your left.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (kard.isSelected())
        if (m_rules.isLegalPlay(m_players[leftPlayerIndex()].hand(), kard.card(), m_players[currentPlayerIndex]))
        {
            cardPlayed(kard.card());
            updateTable();
        }
}

void OldMaidInterface::playerCardMoved()
{
    int size = m_players.size();

    for (int index = 0; index < size; ++index)
    {
        m_players[index].hand().clear();
        for (int i = 0, size = m_playersHands[index]->cardSequence().size(); i < size; ++i)
            m_players[index].hand().addCard(m_playersHands[index]->cardSequence()[i]);
    }
}

void OldMaidInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty() || m_rules.isGameOver(m_players)) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (event->timerId() == m_computerTimerId)
    {
        int size = m_players.size() - 1;
        int index = 0;
        for (; index < size; ++index)
            if (m_players[index].isTurn())
                break;
        if (index < size)
            computerPlay(index);
        else
            QWidget::timerEvent(event);
    }
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void OldMaidInterface::computerPlay(int playerIndex)
{
    OldMaidPlayer *compPlayer=static_cast<OldMaidPlayer *>(&m_players[playerIndex]);

    if (m_players[playerIndex].isTurn())
        m_playersHands[leftPlayerIndex()]->setSelected(m_players[leftPlayerIndex()].hand()[compPlayer->selectCardIndex(m_players[leftPlayerIndex()].hand().size())]);
}
