/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef WAR_H_
#define WAR_H_

#include "gamebase.h"
#include "carddeck.h"
#include "warrules.h"
#include "cardsequence.h"

class ComputerPlayer;
class Player;
class WarInterface;
class UserProfileDatabase;
class Card;

/**
 * This is the logic to the game of war.
 *
 * @author John Schneiderman
 */
class War: public GameBase
{
public:
    /**
     * The default constructor.
     * @param gameInterface is the interface to the game of war.
     * @param profileDatabase is the database of the user's profile.
     */
    War(WarInterface *gameInterface, UserProfileDatabase &profileDatabase);
    /**
     * Default destructor.
     */
    virtual ~War();
    /**
     * Starts to play the game of war.
     * @throw KardsGTError if wrong number of players are added.
     */
    virtual void startPlay();
    /**
     * Saves our current game of war.
     * @param filename is the name of the file to save.
     * @return true if we're successful in saving our game, false if we fail.
     */
    virtual bool save(const QString &filename);
    /**
     * Loads a saved game of war.
     * @param filename is the name of the file to load.
     * @return true if we're successful in loading our game, false if we fail.
     */
    virtual bool load(const QString &filename);
    /**
     * Gives the minimum number of players needed for the game.
     * @return the minimum number of players.
     */
    virtual int minimumPlayers() const;
    /**
     * Gives the maximum number of players allowed for the game.
     * @return the maximum number of players.
     */
    virtual int maximumPlayers() const;

protected:
    /// @param m_rules is the rules for the game of war.
    WarRules m_rules;
    /// @param m_warCards are the cards that each player places in the event of a war.
    CardSequence m_warCards;
    /// @param m_firstToPlay is the index of the player who played the first card in the play sequence.
    int m_firstToPlay;

    /**
     * Handles what to do when a player plays a card.
     * @param card is the card that the player has played.
     */
    virtual void cardPlayed(const Card &card);
    /**
     * Deals out the cards to the players.
     */
    virtual void deal();
    /**
     * Disables the current player, and enables the next player's turn.
     */
    virtual void setupNextPlayer();
    /**
     * Determines and handles if we've come to the end of our game.
     * @return not used.
     */
    virtual int handleGameStatus();

private:
    /// @param m_deck is the card deck for the game.
    CardDeck m_deck;
    /// @param m_inWar indicates if we're currently in a state of war, true if we are, false if we're not.
    bool m_inWar;
    /// @param m_pInterface is the interface to our graphical interface.
    WarInterface *m_pInterface;

    /**
     * Handles the player when he wins a war.
     * @param player is the player who has won the war.
     */
    void winCards(Player &player);
    /**
     * Setups the war requirements for a war scenario.
     */
    void handleWar();
    /**
     * Takes the required number of cards from a players hand and places them in the war pile.
     * @param player is the player to discard the cards to the war pile.
     */
    void discardToWarPile(Player &player);
};
#endif
