/****************************************************************************
** ui.h extension file, included from the uic-generated form implementation.
**
** If you want to add, delete, or rename functions or slots, use
** Qt Designer to update this file, preserving your code.
**
** You should not define a constructor or destructor in this file.
** Instead, write your code in functions called init() and destroy().
** These will automatically be called by the form's constructor and
** destructor.
*****************************************************************************/
#include <string>
using std::string;

/// @param fileFilter is the filter for saved games for KardsGT.
const QString KardsGTInterfaceBase::fileFilter = tr("KardsGT game files (*.kgf)\n" "All files (*)");
/// @param VERSION is the current version of our programme. @note Defined in main.cpp
extern const string VERSION;
/// @param COPYRIGHT is the current copyright years. @note Defined in main.cpp
extern const string COPYRIGHT_YEARS;

/**
 * Default constructor code.
 */
void KardsGTInterfaceBase::init()
{
    assistant=new QAssistantClient("");
    player=new Player();
    playerFactory=new PlayerFactory();
    gameFactory=new GameFactory();
    loading=false;
}

/**
 * Default destructor code.
 */
void KardsGTInterfaceBase::destroy()
{
    delete playerFactory;
    delete gameFactory;
    if (player != NULL)
        delete player;
    if (assistant != NULL)
        delete assistant;
}

/**
 * This displays our game handbook. \n
 * @note the first line is for developers, and the second is for users. Un-comment the appropriate one before compiling.
 */
void KardsGTInterfaceBase::helpHandbook()
{
    assistant->showPage("src/doc/toc.html");
//     assistant->showPage(QString("/usr/share/doc/kardsgt-%1/toc.html").arg(VERSION));
}

/**
 * This starts up the game in our game listing.
 */
void KardsGTInterfaceBase::startGame()
{
    int size=playerFactory->players().size();

    for (int i=0; i < size; ++i)
        gameFactory->game().addOpponent(*playerFactory->players()[i]);
    gameFactory->game().addPlayer(*player);
    if (! loading)
        gameFactory->game().startPlay();
    setActions(true);
}

/**
 * This closes the current game we're playing.
 */
void KardsGTInterfaceBase::closeOpenGame()
{
    if (gameFactory->hasGame())
    {
        if (player != NULL)
            player->clear();
        setActions(false);
        layout->raiseWidget(0);
        if (playerFactory->players().size() > 0)
            playerFactory->clear();
        dynamic_cast<QWidget *>(&gameFactory->game())->close();
        gameFactory->clear();
        gamePlayed = "";
        if (! loading)
            opponents.clear(); // Handled by the loadGame
    }
}

/**
 * This gives the information about our game.
 */
void KardsGTInterfaceBase::helpAbout()
{
    QMessageBox::about(this, tr("About KardsGT"), tr(
                           QString("<h2>KardsGT version %1</h2>"
                                   "<p>Copyright &copy; %2 John Schneiderman"
                                   "<p>KardsGT comes with ABSOLUTELY NO WARRANTY. <br>This is free software, and you are welcome to redistribute it under certain conditions; see the COPYING file for details, or the Free Software Foundation's GPL.").arg(VERSION).arg(COPYRIGHT_YEARS)));
}

/**
 * This gives the information about QT.
 */
void KardsGTInterfaceBase::helpAboutQT()
{
    QMessageBox::aboutQt(this, tr("KardsGT"));
}

/**
 * This starts a new game.
 * @throw KardsGTError if we cannot determine what game we're trying to restart.
 */
void KardsGTInterfaceBase::newGame()
{
    if (gamePlayed == "cribbage")
    {
        closeOpenGame();
        turnOnCribbage();
    }
    else if (gamePlayed == "war")
    {
        closeOpenGame();
        turnOnWar();
    }
    else if (gamePlayed == "euchre")
    {
        closeOpenGame();
        turnOnEuchre();
    }
    else if (gamePlayed == "hearts")
    {
        closeOpenGame();
        turnOnHearts();
    }
    else if (gamePlayed == "old maid")
    {
        closeOpenGame();
        turnOnOldMaid();
    }
    else if (gamePlayed == "spades")
    {
        closeOpenGame();
        turnOnSpades();
    }
    else
        throw KardsGTError("KardsGTInterfaceBase", "newGame", "Failed to find what game we're trying to restart!");
}

/**
 * This saves the current game we're playing.
 */
void KardsGTInterfaceBase::saveGame()
{
    QString filename = QFileDialog::getSaveFileName(
                           ".",
                           fileFilter,
                           this,
                           "save file dialogue",
                           "Choose a name to save game under..." );
    if (! filename.isEmpty())
        if (filename.endsWith(".kgf") || filename.contains('.'))
        {
            if (! gameFactory->game().save(filename))
                QMessageBox::critical(this, "Save Error", QString("The game file %1 failed to save!").arg(filename));
        }
        else
        {
            if (! gameFactory->game().save(filename + ".kgf"))
                QMessageBox::critical(this, "Save Error", QString("The game file %1 failed to save!").arg(filename + ".kgf"));
        }
}

/**
 * This loads a previously played game.
 * @throw KardsGTError if we cannot identify the type of game file we're trying to load.
 */
void KardsGTInterfaceBase::loadGame(QString filename)
{
    int numberOfPlayers=0;
    int playersFound=1; // We always have a human player

    try
    {
        if (! filename.isEmpty())
        {
            QFile file(filename);
            QTextStream in(&file);
            QString garbage, gameName;

            if (! file.open(IO_ReadOnly | IO_Translate))
            {
                QMessageBox::critical(this, "Load Error", QString("The game file %1 failed to load!").arg(filename));
                file.close();
                return;
            }
            in.setEncoding(QTextStream::UnicodeUTF8);
            in >> garbage >> gameName;
            gameName.replace("_", " ");
            selectGameAction(gameName);
            if (gamePlayed == "unknown")
                throw KardsGTError("KardsGTInterfaceBase", "loadGame", "Failed to identify type of game file!");
            loading=true;
            opponents.clear();
            in >> garbage; //Read players label
            in >> garbage >> numberOfPlayers;
            do // Find the AI players
            {
                in >> garbage;
                if (garbage == "Name:")
                {
                    in >> garbage; // Should be players name.
                    opponents.push_back(garbage.lower());
                    playersFound++;
                }
            }
            while (numberOfPlayers != playersFound);
            while (true) // Find the human player's name
            {
                in >> garbage;
                if (garbage == "Name:")
                    break;
            };
            in >> garbage;
            player->setName(garbage); // Set the human players name.
            file.close();
            newGame();
            if (gameFactory->hasGame() && (! gameFactory->game().load(filename)))
            {
                QMessageBox::critical(this, "Load Error", QString("The game file %1 failed to load!").arg(filename));
                closeOpenGame();
            }
            loading=false;
        }
    }
    catch (KardsGTError error)
    {
        std::cerr << error << std::endl;
        QMessageBox::critical(this, "Load Error", error.errorMessage());
        closeOpenGame();
    }
}

/**
 * This sets the game menu actions to their appropriate states.
 * @param actionOn is true if you want to turn all the game actions on, false elsewise.
 */
void KardsGTInterfaceBase::setActions(bool actionOn)
{
    // En/disable file operations.
    gameNewAction->setEnabled(actionOn);
    gameSaveAction->setEnabled(actionOn);
    gameCloseAction->setEnabled(actionOn);
    // En/disable games available.
    gameSelectCribbageAction->setEnabled(! actionOn);
    gameSelectEuchreAction->setEnabled(! actionOn);
    gameSelectHeartsAction->setEnabled(! actionOn);
    gameSelectOld_MaidAction->setEnabled(! actionOn);
    gameSelectSpadesAction->setEnabled(! actionOn);
    gameSelectWarAction->setEnabled(! actionOn);
    // Toggle on/off the game we're playing
    if ((! actionOn) && gameSelectCribbageAction->isOn())
        gameSelectCribbageAction->setOn(actionOn);
    if ((! actionOn) && gameSelectEuchreAction->isOn())
        gameSelectEuchreAction->setOn(actionOn);
    if ((! actionOn) && gameSelectHeartsAction->isOn())
        gameSelectHeartsAction->setOn(actionOn);
    if ((! actionOn) && gameSelectOld_MaidAction->isOn())
        gameSelectOld_MaidAction->setOn(actionOn);
    if ((! actionOn) && gameSelectWarAction->isOn())
        gameSelectWarAction->setOn(actionOn);
    if ((! actionOn) && gameSelectSpadesAction->isOn())
        gameSelectSpadesAction->setOn(actionOn);
}

/**
 * Turns on the game being played.
 * @param game is the name of the game we want to turn on.
 */
void KardsGTInterfaceBase::selectGameAction(const QString &game)
{
    if (game == "cribbage")
    {
        gameSelectCribbageAction->setOn(true);
        gamePlayed="cribbage";
    }
    else if (game == "war")
    {
        gameSelectWarAction->setOn(true);
        gamePlayed="war";
    }
    else if (game == "hearts")
    {
        gameSelectHeartsAction->setOn(true);
        gamePlayed="hearts";
    }
    else if (game == "old maid")
    {
        gameSelectOld_MaidAction->setOn(true);
        gamePlayed="old maid";
    }
    else if (game == "spades")
    {
        gameSelectSpadesAction->setOn(true);
        gamePlayed="spades";
    }
    else if (game == "euchre")
    {
        gameSelectEuchreAction->setOn(true);
        gamePlayed="euchre";
    }
    else
        gamePlayed="unknown";
}

/**
 * An abstract method to setup the game we've selected.
 */
void KardsGTInterfaceBase::setupGame()
{}

/**
 * Loads the user selected game.
 */
void KardsGTInterfaceBase::loadGame()
{
    loadGame(QFileDialog::getOpenFileName(
                 ".",
                 fileFilter,
                 this,
                 "load file dialogue",
                 "Choose a game to load..." ));
}

/**
 * This sets up and starts a Cribbage game.
 */
void KardsGTInterfaceBase::turnOnCribbage()
{
    selectGameAction("cribbage");
    setupGame();
}

/**
 * This sets up and starts a War game.
 */
void KardsGTInterfaceBase::turnOnWar()
{
    selectGameAction("war");
    setupGame();
}

/**
 * This sets up and starts a game of Hearts.
 */
void KardsGTInterfaceBase::turnOnHearts()
{
    selectGameAction("hearts");
    setupGame();
}

/**
 * This sets up and starts a game of Spades.
 */
void KardsGTInterfaceBase::turnOnSpades()
{
    selectGameAction("spades");
    setupGame();
}


/**
 * This sets up and starts a game of Old Maid.
 */
void KardsGTInterfaceBase::turnOnOldMaid()
{
    selectGameAction("old maid");
    setupGame();
}

void KardsGTInterfaceBase::turnOnEuchre()
{
    selectGameAction("euchre");
    setupGame();
}

/**
 * This when implemented adjusts the settings for KardsGT.
 */
void KardsGTInterfaceBase::settingsConfigureKardsGT()
{}
