/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "normaneuchreai.h"
#include "rulebase.h"
#include "cardproperties.h"
#include "../../games/euchre/euchrerules.h"

NormanEuchreAI::NormanEuchreAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_pRules = &rules;
    m_hand = hand;
}

NormanEuchreAI::~NormanEuchreAI()
{}

int NormanEuchreAI::turnUp(const Card &turnUpCard, int score, int opponentScore) const
{
    if ((score == 4) && (opponentScore > 0)) // Prevent opponent's from winning by marching
        return EuchreRules::WANT_TURN_UP;
    if (basicStrategies.numberOfCardsHigher(m_hand, Card::QUEEN) >= 3)
        return EuchreRules::WANT_TURN_UP;
    if (basicStrategies.numberOfCardsOfSuit(m_hand, turnUpCard.suit()) >= 2)
        return EuchreRules::WANT_TURN_UP;
    return EuchreRules::DOES_NOT_DECIDE;
}

int NormanEuchreAI::selectTrump(const Card &turnUpCard, int, int) const
{
    for (int suit = 0; suit < 4; ++suit)
        if (static_cast<Card::Suit>(suit) != turnUpCard.suit())
            if (basicStrategies.numberOfCardsOfSuit(m_hand, static_cast<Card::Suit>(suit)) >= 2)
                return suit;
    return Card::SUIT_ERR;
}

int NormanEuchreAI::playAlone(int score, int opponentScore) const
{
    if ((score > 0) && (opponentScore == 4)) // Try to win with a march
        return EuchreRules::PLAY_ALONE;
    if (basicStrategies.numberOfCardsHigher(m_hand, Card::QUEEN) >= 5)
        return EuchreRules::PLAY_ALONE;
    return EuchreRules::DOES_NOT_DECIDE;
}

int NormanEuchreAI::defendAlone() const
{
    if (basicStrategies.numberOfCardsHigher(m_hand, Card::QUEEN) >= 5)
        return EuchreRules::DEFEND_ALONE;
    return EuchreRules::DOES_NOT_DECIDE;
}

int NormanEuchreAI::wantTurnUp() const
{
    // The "m_playSequence" should actually be the card from the deck
    if (m_pRules->rankValue(m_playSequence.front()) >= Card::TEN)
        return EuchreRules::WANT_TURN_UP_CARD;
    return EuchreRules::DOES_NOT_DECIDE;
}

CardSequence NormanEuchreAI::selectCards(const Card &partnersCard) const
{
    bool selectedCard = false;
    CardSequence selection, testSelection;
    CardProperties handProperties(m_hand);
    CardProperties playSequenceProperties(m_playSequence);
    Card::Suit trumpSuit = static_cast<const EuchreRules *>(m_pRules)->trumpSuit();

    // If leading, then lead with the highest card in our hand
    if (m_playSequence.isEmpty())
    {
        selection = basicStrategies.highestCard(m_hand);
        selectedCard = true;
    }

    // Make sure not to play over partner if parnter leads, then play lowest
    if ((! selectedCard))
    {
        Card highestCard = basicStrategies.highestCardOfSuit(m_playSequence, m_playSequence.front().suit());

        // Play the lowest legal card
        if (highestCard == partnersCard)
        {
            selectedCard = true;
            selection = basicStrategies.lowestCardOfSuit(m_hand, highestCard.suit());
            if (selection.isEmpty())
            {
                // If trump was lead we have to check to see if we have the left-bower.
                if (m_hand.hasCard(Card(Card::JACK, static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit())))
                {
                    selection.clear();
                    selection.addCard(Card(Card::JACK, static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit()));
                }
                else
                    selection = basicStrategies.lowestCard(m_hand);
            }
        }
    }

    // If lead suit is trump try to beat the highest trump, else play the lowest
    if ((! selectedCard) && (m_playSequence.front().suit() == trumpSuit))
    {
        testSelection = basicStrategies.highestCardOfSuit(m_hand, m_playSequence.front().suit()); // Suit is trump
        Card highestCard = basicStrategies.highestCardOfSuit(m_playSequence, m_playSequence.front().suit());

        // If our highest card it's not the right bower, see if we have the left bower
        if (m_hand.hasCard(Card(Card::JACK, static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit()))) // Do we have the left bower
            if (((! testSelection.isEmpty()) && (testSelection.front().rank() != Card::JACK)) || testSelection.isEmpty())
            {
                testSelection.removeCard(testSelection.front());
                testSelection.addCard(Card(Card::JACK, static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit()));
            }
        if ((! testSelection.isEmpty()) && (m_pRules->rankValue(testSelection.front()) > m_pRules->rankValue(highestCard))) // Make sure we'll beat the highest card
        {
            selection = testSelection;
            selectedCard = true;
        }
        else // Play our lowest trump card
        {
            testSelection = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
            if (! testSelection.isEmpty())
            {
                selection = testSelection;
                selectedCard = true;
            }
            else if (m_hand.hasCard(Card(Card::JACK, static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit()))) // Do we have the left bower
            {
                selection.addCard(Card(Card::JACK, static_cast<const EuchreRules *>(m_pRules)->leftBowerSuit()));
                selectedCard = true;
            }
        }
    }

    // If trump hasn't been played, then play the highest card of the lead suit
    if ((! selectedCard) && playSequenceProperties.suits(trumpSuit).isEmpty())
    {
        testSelection = basicStrategies.highestCardOfSuit(m_hand, m_playSequence.front().suit());
        Card highestCard = basicStrategies.highestCardOfSuit(m_playSequence, m_playSequence.front().suit());

        if (testSelection.isEmpty()) // If don't have a card of the lead suit, then play a trump
        {
            testSelection = basicStrategies.highestCardOfSuit(m_hand, trumpSuit);
            if (! testSelection.isEmpty())
            {
                selection = testSelection;
                selectedCard = true;
            }
        }
        else if (m_pRules->rankValue(testSelection.front()) > m_pRules->rankValue(highestCard)) // Make sure we'll beat the highest card
        {
            selection = testSelection;
            selectedCard = true;
        }
        else // Play our lowest card
        {
            selection = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
            selectedCard = true;
        }
    }
    else if (! selectedCard)// If trump has been played, then play a low card of the lead suit
    {
        testSelection = basicStrategies.lowestCardOfSuit(m_hand, m_playSequence.front().suit());
        // If don't have a card of the lead suit, then play a higher trump
        if (testSelection.isEmpty())
        {
            testSelection = basicStrategies.highestCardOfSuit(m_hand, trumpSuit);
            Card highestTrumpCard = basicStrategies.highestCardOfSuit(m_playSequence, trumpSuit);

            if (! testSelection.isEmpty()) // Make sure we have a trump
            {
                // See if we have a higher trump
                if (m_pRules->rankValue(testSelection.front()) > m_pRules->rankValue(highestTrumpCard))
                {
                    selection = testSelection;
                    selectedCard = true;
                }
                else // Play the lowest non-trump card
                {
                    testSelection = basicStrategies.lowestNonTrumpCard(m_hand, trumpSuit);
                    if (! testSelection.isEmpty()) // Make sure we have a trump
                    {
                        selection = testSelection;
                        selectedCard = true;
                    }
                }
            }
        }
        else // Play the lowest card of the lead suit.
        {
            selection = testSelection;
            selectedCard = true;
        }
    }

    // Play the lowest card from a suit
    if (! selectedCard)
        selection = basicStrategies.lowestCard(m_hand);
    return selection;
}

CardSequence NormanEuchreAI::discard() const
{
    return basicStrategies.lowestNonTrumpCard(m_hand, static_cast<const EuchreRules *>(m_pRules)->trumpSuit());
}

