/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "cardpile.h"
#include "kardsgterror.h"

#include <qstring.h>
#include <qtextstream.h>

CardPile::CardPile(): m_pile()
{}

CardPile::CardPile(const CardSequence &sequence)
{
    for (int i=0; i < sequence.size(); ++i)
        m_pile.addCard(sequence[i]);
}

void CardPile::clear()
{
    m_pile.clear();
}

const Card& CardPile::topCard() const
{
    return m_pile.front();
}

int CardPile::size() const
{
    return m_pile.size();
}

const Card& CardPile::bottomCard() const
{
    return m_pile.back();
}

void CardPile::addCard(const Card &card)
{
    m_pile.addCardToFront(card);
}

Card CardPile::removeCard()
{
    Card topCard;

    topCard=m_pile.front();
    if (! m_pile.removeCard(topCard))
    	throw KardsGTError("CardPile", "removeCard", "Failed to remove the top card!");
    return topCard;
}

void CardPile::addCardToBottom(const Card &card)
{
    m_pile.addCard(card);
}

Card CardPile::removeCardFromBottom()
{
    Card bottomCard;

    bottomCard=m_pile.back();
    if (! m_pile.removeCard(bottomCard))
    	throw KardsGTError("CardPile", "removeCardFromBottom", "Failed to remove the bottom card!");
    return bottomCard;
}

bool CardPile::operator==(const CardPile &rhs)
{
    int size=m_pile.size();

    if (size != rhs.size())
        return false;
    for (int i=0; i < size; i++)
        if (m_pile[i] != rhs.m_pile[i])
            return false;
    return true;
}

CardPile& CardPile::operator+=(const CardPile& rhs)
{
    int pileSize=rhs.m_pile.size();

    if (this != &rhs)
        for (int i=0; i < pileSize; i++)
            m_pile.addCard(rhs.m_pile[i]);
    return *this;
}

bool CardPile::isEmpty() const
{
    if (m_pile.size() == 0)
        return true;
    else
        return false;
}

ostream& operator<<(ostream &out, const CardPile &cardPile)
{
    int pileSize=cardPile.m_pile.size();

    for (int i=0; i < pileSize; i++)
        if (i == (pileSize - 1))
            out << cardPile.m_pile[i];
        else
            out << cardPile.m_pile[i] << " ";
    return out;
}

istream& operator>>(istream &in, CardPile &cardPile)
{
    Card card;

    while (!in.fail())
    {
        in >> card;
        cardPile.m_pile.addCard(card);
    }
    return in;
}

QTextStream& operator<<(QTextStream &out, const CardPile &pile)
{
    int pileSize=pile.m_pile.size();

    for (int i=0; i < pileSize; i++)
        out << pile.m_pile[i] << " ";
    out << Card();
    return out;
}

QTextStream& operator>>(QTextStream &in, CardPile &pile)
{
    Card card;

    pile.clear();
    in >> card;
    while (! card.isEmpty())
    {
        pile.m_pile.addCard(card);
        in >> card;
    }
    return in;
}
