/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef KARDSGTERROR_H
#define KARDSGTERROR_H

#include <iostream>
using std::ostream;
#include <stdexcept>
using std::runtime_error;

class QString;

/**
 * This is the catch all run time error handling class for KardsGT.
 *
 * @author John Schneiderman
 */
class KardsGTError: public runtime_error
{
public:
    /**
     * Default constructor
     * @param name is the name of the class generating the error.
     * @param method is the name of the method generating the error.
     * @param message is the message explaining what cause the error.
     */
    KardsGTError(const char * name, const char * method, const char * message);
    /**
     * Default constructor
     * @param name is the name of the class generating the error.
     * @param method is the name of the method generating the error.
     * @param message is the message explaining what cause the error.
     */
    KardsGTError(QString name, QString method, QString message);
    /**
     * Copy constructor
     * @param kardsGTError is the what to copy into *this.
     */
    KardsGTError(const KardsGTError &kardsGTError);
    /**
     * Destructor
     */
    ~KardsGTError() throw();
    /**
     * Gives the name of the offending class.
     * @return the name of the class that generated the exception.
     */
    QString className() const;
    /**
     * Gives the name of the method the caused the exception.
     * @return the name of the method in the class.
     */
    QString classMethod() const;
    /**
     * Gives the error message.
     * @return the error message that caused the exception.
     */
    QString errorMessage() const;
    /**
     * Stream insertion operator to display the thrown error.
     * @param out is the buffer to write the error message to in the form of: class, method: errormessage.
     * @param error is the error thrown.
     * @return out.
     */
    friend ostream& operator<<(ostream &out, const KardsGTError &error);
    /**
     * Equality operator.
     * @note Mostly used for testing the class.
     * @param error this is the error to test.
     * @return true if class name, method name and error message are all the same.
     */
    bool operator==(const KardsGTError &error) const;

private:
    /// @param m_pName is the name of the class that caused the error.
    QString *m_pName;
    /// @param m_pMethod is the name of the method that caused the error.
    QString *m_pMethod;
    /// @param m_pMessage is the error message.
    QString *m_pMessage;
};
#endif
