/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef CRIBBAGERULES_H
#define CRIBBAGERULES_H

#include "rulebase.h"

#include <vector>
using std::vector;

class CardSequence;
class Player;
class PlayerList;

/**
 * This is the rules for the game of cribbage.
 *
 * @author John Schneiderman
 */
class CribbageRules:public RuleBase
{
public:
    /**
     * This is how many points for having types of jacks.
     * @param HEELS is for when you have a Jack on the starter card.
     * @param NOBS is for when you have a Jack of the same suit as the starter card.
     */
    enum ScoreJack { HEELS=2, NOBS=1 };
    /**
     * This is how many points for having certain combinations.
     * @param FIFTEEN is when you have cards that equal to 15.
     * @param THIRTY_ONE is when you have cards that equal 31.
     */
    enum ScoreCombinations { FIFTEEN=2, THIRTY_ONE=2 };
    /**
     * This is how many points for having certain conditions while in play.
     * @param GO is when your opponent cannot play any of his cards without going over 31.
     * @param LAST is when you play the last card.
     */
    enum ScoreInPlay { GO=1, LAST=1 };
    /**
     * This is how many points for having special types of pairs.
     * @param PAIR is when you have two cards of the same rank.
     * @param PAIR_THREE is when you have three cards of the same rank.
     * @param PAIR_FOUR is when you have four cards of the same rank.
     * @param PAIR_ERR is when there is an error in paring.
     */
    enum ScorePairs { PAIR=2, PAIR_THREE=6, PAIR_FOUR=12, PAIR_ERR=-1 };
    /**
     * This is how many points for having special types of runs.
     * @param RUN is when you have 3 or more cards that are in sequential order. KQJ
     * @param RUN_DOUBLE is when you have 4 or more cards that are in sequential order with a repeated card. KQQJ
     * @param RUN_DOUBLE_FOUR is when you have 5 cards that are in sequential order with a repeated card. KQQJ10
     * @param RUN_TRIPLE is when you have 5 cards that are in sequential order with 2 repeated cards. KQQQJ
     * @param RUN_QUADRUPLE is when you have 5 cards that are in sequential order with 2 sets of repeated cards. KQQJJ
     * @param RUN_ERR is when a run cannot be determined.
     */
    enum ScoreRuns { RUN=1, RUN_DOUBLE=8, RUN_DOUBLE_FOUR=10, RUN_TRIPLE=15, RUN_QUADRUPLE=16, RUN_ERR=-1 };
    /**
     * This is how many points for having special types of flushes.
     * @param FLUSH_FOUR is when you have four cards of the same suit.
     * @param FLUSH_FIVE is when you have five cards of the same suit.
     */
    enum ScoreFlush { FLUSH_FOUR=4, FLUSH_FIVE=5 };
    /**
     * These are some constants that can define the course of the game.
     * @param MINIMUM_PLAYERS is the minimum number of players for the game.
     * @param MAXIMUM_PLAYERS is the maximum number of players for the game.
     * @param NUMBER_CARDS_TO_CRIB is the number of cards to give to the crib.
     * @param WINNING_ROUND_SCORE is the score needed to win a round.
     * @param MAXIMUM_CARDS_DEALT is the number of cards to deal each rounds.
     * @param MAXIMUM_PHASE_POINTS is the maximum score that a single phase can reach.
     * @param NUMBER_OF_STAKES is the number of stakes to play the game for.
     * @param SKUNK is the score that is needed for the other player to be skunked.
     * @param LURCHED is the score that is needed for the other player to be lurched, or "double skunked".
     * @param CARDS_LEFT_IN_HAND is the number of cards left in the hand after passing to the crib.
     */
    enum PublicConstants { MINIMUM_PLAYERS=2, MAXIMUM_PLAYERS=2, NUMBER_CARDS_TO_CRIB=2, WINNING_ROUND_SCORE=121, MAXIMUM_CARDS_DEALT=6, MAXIMUM_PHASE_POINTS=31, NUMBER_OF_STAKES=7, SKUNK=90, LURCHED = 60, CARDS_LEFT_IN_HAND=4 };

    /**
     * Default constructor.
     */
    CribbageRules();
    /**
     * Gives the maximum number of players that cribbage can have.
     * @return the maximum players.
     */
    virtual int maximumNumberOfPlayers() const;
    /**
     * Gives the minimum number of players that cribbage needs.
     * @return the minimum players.
     */
    virtual int minimumNumberOfPlayers() const;
    /**
     * Gives the score that a player needs to win the game.
     * @return the number of stakes the player must win.
     */
    virtual int winningGameScore() const;
    /**
     * Gives the number of cards to deal to each player for a particular set.
     * @return the number of cards.
     */
    virtual int numberOfCardsToDeal(int numberOfPlayers=-1) const;
    /**
     * Gives how much a particular card is worth in cribbage.
     * @param card is the card to examine.
     * @return how many points a card is worth.
     */
    virtual int cardValue(const Card &card) const;
    /**
     * This determines if a card that a player wants to play in cribbage is allowed.
     * @param sequence is the cards that have been played so far in the round.
     * @param cardPlayed is the card that the player wants to play.
     * @param player is the player who wants to play the card.
     * @return true if the card is allowed false elsewise.
     */
    virtual bool isLegalPlay(const CardSequence &sequence, const Card &cardPlayed, const Player &player=Player()) const;
    /**
     * This determines if the game of cribbage is over.
     * @param players are all of the players playing cribbage.
     * @return true if the one of the players has meet the conditions to win the game, false elsewise.
     */
    virtual bool isGameOver(const PlayerList &players) const;
    /**
     * Determines if a player has met the conditions to win the round.
     * @param players are all of the players playing cribbage.
     * @return true if the one of the players has meet the conditions to win the game, false elsewise.
     */
    virtual bool isRoundOver(const PlayerList &players) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual bool isPhaseOver(const PlayerList &, const CardSequence &) const;
    /**
     * Rearranges card sequences to fit Cribbage special pair types.
     * @param pairs is the card sequences to rearrange to fit the Cribbage special pair types.
     */
    void createPairTypes(vector<CardSequence> &pairs) const;
    /**
     * Rearranges card sequences to fit Cribbage special pair types.
     * @param pairs is the card sequence to examine for specail pair types.
     * @return the card sequences found that had the special pair types.
     */
    vector<CardSequence> createPairTypes(const vector<CardSequence> &pairs) const;
    /**
     * Gives the type of pair that is found in a sequence of cards.
     * @param pair is the sequences of cards that contains the pair.
     * @return the type of pair found, gives PAIR_ERR if the type cannot be found.
     */
    ScorePairs pairType(const CardSequence &pair) const;
    /**
     * Rearranges card sequences to fit Cribbage special run types.
     * @param runs is the card sequences to rearrange to fit the Cribbage special run types.
     */
    void createRunTypes(vector<CardSequence> &runs) const;
    /**
     * Rearranges card sequences to fit Cribbage special run types.
     * @param runs is the card sequence to examine for special run types.
     * @return the card sequences found that had the special run types.
     */
    vector<CardSequence> createRunTypes(const vector<CardSequence> &runs) const;
    /**
     * Gives the type of run that is found in a sequence of cards.
     * @param runs are the sequences of cards that contain a run.
     * @return the type of run found, gives RUN_ERR if the run type cannot be found.
     */
    ScoreRuns runType(const vector<CardSequence> &runs) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual int rankValue(const Card &) const;

private:
    /**
     * How much each card is worth.
     * @param RANK_ERR is if there is a problem determining the rank value.
     */
    enum CardValues {ACE=1, TWO=2, THREE=3, FOUR=4, FIVE=5, SIX=6, SEVEN=7, EIGHT=8, NINE=9, TEN=10, JACK=10, QUEEN=10, KING=10, RANK_ERR=-1};
};
#endif
