/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "kardsgtinterface.h"
#include "kardsgterror.h"

#include <qapplication.h>
#include <iostream>
using std::cout;
using std::cerr;
using std::endl;
#include <string>
using std::string;
#include <unistd.h>
#include <cstdlib>

/// @param VERSION is the current version of our programme.
extern const string VERSION="0.6.3";
/// @param COPYRIGHT is the current copyright years of our programme.
extern const string COPYRIGHT_YEARS="2005 - 2007";

/**
 * This is the entry to start our programme.
 *
 * @param argc is the number of arguments given at the command line.
 * @param argv is the array containing the arguments. The arguments available are: \n
 * -v for the version information. \n
 * -h for a short help information. \n
 * -u path  This will set the user profile location. The path only needs to be to the top level directory containing the database. \n
 * -l game  This will load a game file, where game is a KardsGT game file.
 * @return whether the programme exited successfully or not.
 */
int main(int argc, char * argv[])
{
    char processedArugments;
    string gameToLoad="", userProfilePath=".";

    while ((processedArugments=getopt(argc, argv, "vhu:l:")) != -1)
    {
        switch (processedArugments)
        {
        case 'h':
            cout << "usage: kardsgt [options]" << endl;
            cout << "Options:" << endl;
            cout << "  -v                       Shows the version information." << endl;
            cout << "  -l <game>                Loads a game KardsGT game." << endl;
            cout << "  -u <path>                Changes the user profile path location." << endl;
            cout << endl << "For more information see the man page." << endl;
            cout << "For bug reporting instructions, please see the man page." << endl;
            return EXIT_SUCCESS;
        case 'v':
            cout << "KardsGT version " << VERSION << endl;
            cout << "Copyright (C) "<< COPYRIGHT_YEARS << " John Schneiderman" << endl;
            return EXIT_SUCCESS;
        case 'l':
            gameToLoad=optarg;
            break;
        case 'u':
            userProfilePath=optarg;
            break;
        default:
            cerr << "Unrecognised option encountered -" << static_cast<char>(optopt) << endl;
            return EXIT_FAILURE;
        }
    }

    // Display our copyright notice.
    cout << "KardsGT version " << VERSION << endl;
    cout << "Copyright (C) "<< COPYRIGHT_YEARS << " John Schneiderman" << endl;
    cout << "KardsGT comes with ABSOLUTELY NO WARRANTY." << endl;
    cout << "This is free software, and you are welcome to redistribute it" << endl;
    cout << "under certain conditions; see the COPYING file for details," << endl;
    cout << "or the Free Software Foundation's GPL." << endl << endl;

    // Launch the game.
    try
    {
        QApplication application(argc, argv);
        KardsGTInterface *mainWindow=new KardsGTInterface(userProfilePath, 0, "KardsGT", gameToLoad);

        mainWindow->show();
        application.connect(&application, SIGNAL(lastWindowClosed()), &application, SLOT(quit()));
        return application.exec();
    }
    catch (KardsGTError error)
    {
        cerr << error << endl;
        return EXIT_FAILURE;
    }
    catch (...)
    {
        cerr << "Unknown error occurred..." << endl;
        return EXIT_FAILURE;
    }
}
