/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "sally.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
// Games Sally knows how to play
#include "sallycribbageai.h"
#include "sallyheartsai.h"
#include "sallyspadesai.h"
#include "sallyeuchreai.h"

Sally::Sally(): AIBase()
{}

Sally::~Sally()
{}

CardSequence Sally::whichCards(const QString& game, const CardSequence& playSequence, const RuleBase& rules, const CardSequence& hand, const GameInformation &specialInformation) const
{
    CardSequence cards;

    if (game.contains("cribbage"))
        cards = handleCribbageCards(game, playSequence, rules, hand);
    else if (game.contains("spades"))
        cards = handleSpadesCards(game, playSequence, rules, hand, specialInformation);
    else if (game.contains("hearts"))
        cards = handleHeartsCards(game, playSequence, rules, hand);
    else if (game.contains("euchre"))
        cards = handleEuchreCards(game, playSequence, rules, hand, specialInformation);
    else
    {
        BasicGameStrategies ai(rules);
        cards=ai.selectCards(1, hand, playSequence);
    }
    return cards;
}

int Sally::decision(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    int bid = -1;

    if (game.contains("spades"))
        bid = handleSpadesBid(game, playSequence, rules, hand, specialInformation, score);
    else if (game.contains("euchre"))
        bid = handleEuchreDecisions(game, playSequence, rules, hand, specialInformation, score);
    else
    {
        BasicGameStrategies ai(rules);
        bid=ai.randomBid(1, 100);
    }
    return bid;
}

Player::skill_Level Sally::skillLevel(const QString& game) const
{
    if (game == "spades")
        return Player::Amateur;
    else if (game == "cribbage")
        return Player::Novice;
    else if (game == "hearts")
        return Player::Apprentice;
    else if (game == "old maid")
        return Player::Master;
    else if (game == "euchre")
        return Player::Apprentice;
    else
        return Player::Newb;
}

QString Sally::name() const
{
    return "Sally";
}

CardSequence Sally::handleCribbageCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "cribbage")
    {
        SallyCribbageAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "cribbageDealerCrib")
    {
        SallyCribbageAI ai(playSequence, rules, hand);
        return ai.myCrib();
    }
    else if (game == "cribbageNonDealerCrib")
    {
        SallyCribbageAI ai(playSequence, rules, hand);
        return ai.opponentsCrib();
    }
    throw KardsGTError("Sally", "handleCribbageCards", "Failed to determine game type or phase.");
}

CardSequence Sally::handleSpadesCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "spades")
    {
        SallySpadesAI ai(playSequence, rules, hand);
        Card partnersCard;

        partnersCard.setCard(specialInformation.value("Partner Played"));
        // If we bid nil play differently
        if (specialInformation.value("Made Nil Bid") == "true")
            return ai.selectCardsForNilBid();
        else
            return ai.selectCards(partnersCard);
    }
    else if (game == "spadesExchangeCards")
    {
        SallySpadesAI ai(playSequence, rules, hand);
        return ai.exchangeCards();
    }
    throw KardsGTError("Sally", "handleSpadesCards", "Failed to determine game type or phase.");
}

CardSequence Sally::handleHeartsCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand) const
{
    if (game == "hearts")
    {
        SallyHeartsAI ai(playSequence, rules, hand);
        return ai.selectCard();
    }
    else if (game == "heartsPassPhase")
    {
        SallyHeartsAI ai(playSequence, rules, hand);
        return ai.passCards();
    }
    throw KardsGTError("Sally", "handleHeartsCards", "Failed to determine game type or phase.");
}

int Sally::handleSpadesBid(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "spadesBidDoubleNil")
    {
        SallySpadesAI ai(playSequence, rules, hand);
        return ai.bidDoubleNil(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "spadesBidNil")
    {
        SallySpadesAI ai(playSequence, rules, hand);
        return ai.bidNil(specialInformation.value("Partners Bid").toInt());
    }
    throw KardsGTError("Sally", "handleSpadesBid", "Failed to determine game type or phase.");
}

int Sally::handleEuchreDecisions(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const
{
    if (game == "euchreMakeTurnUp")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        return ai.turnUp(playSequence.front(), score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchreSelectTrump")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        return ai.selectTrump(playSequence.front(), score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchrePlayAlone")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        return ai.playAlone(score, specialInformation.value("Opponents Score").toInt());
    }
    else if (game == "euchreDefendAlone")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        return ai.defendAlone();
    }
    else if (game == "euchreWantsTurnUp")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        return ai.wantTurnUp();
    }
    throw KardsGTError("Sally", "handleEuchreDecisions", "Failed to determine game type or phase.");
}

CardSequence Sally::handleEuchreCards(const QString& game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const
{
    if (game == "euchre")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        Card partnersCard;

        partnersCard.setCard(specialInformation.value("Partner Played"));
        return ai.selectCards(partnersCard);
    }
    else if (game == "euchreDiscard")
    {
        SallyEuchreAI ai(playSequence, rules, hand);
        return ai.discard();
    }
    throw KardsGTError("Sally", "handleEuchreCards", "Failed to determine game type or phase.");
}
