/***************************************************************************
 *   Copyright (C) 2005 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "cribbagerules.h"
#include "cardsequence.h"
#include "player.h"
#include "playerlist.h"
#include "kardsgterror.h"

CribbageRules::CribbageRules(): RuleBase()
{}

int CribbageRules::maximumNumberOfPlayers() const
{
    return MAXIMUM_PLAYERS;
}

int CribbageRules::minimumNumberOfPlayers() const
{
    return MINIMUM_PLAYERS;
}

int CribbageRules::winningGameScore() const
{
    return NUMBER_OF_STAKES;
}

int CribbageRules::numberOfCardsToDeal(int) const
{
    return MAXIMUM_CARDS_DEALT;
}

int CribbageRules::cardValue(const Card &card) const
{
    CardValues value=RANK_ERR;

    switch(card.rank())
    {
    case Card::ACE:
        value=ACE;
        break;
    case Card::TWO:
        value=TWO;
        break;
    case Card::THREE:
        value=THREE;
        break;
    case Card::FOUR:
        value=FOUR;
        break;
    case Card::FIVE:
        value=FIVE;
        break;
    case Card::SIX:
        value=SIX;
        break;
    case Card::SEVEN:
        value=SEVEN;
        break;
    case Card::EIGHT:
        value=EIGHT;
        break;
    case Card::NINE:
        value=NINE;
        break;
    case Card::TEN:
        value=TEN;
        break;
    case Card::JACK:
        value=JACK;
        break;
    case Card::QUEEN:
        value=QUEEN;
        break;
    case Card::KING:
        value=KING;
        break;
    default:
        value=RANK_ERR;
    }
    return value;
}

bool CribbageRules::isLegalPlay(const CardSequence &sequence, const Card &cardPlayed, const Player &) const
{
    int points=0;

    for (int i=0; i < sequence.size(); i++)
        points+=cardValue(sequence[i]);
    if ((points + cardValue(cardPlayed)) > MAXIMUM_PHASE_POINTS)
        return false;
    return true;
}

void CribbageRules::createPairTypes(vector<CardSequence> &pairs) const
{
    CardSequence bin[Card::NUMBER_OF_RANKS];
    int size=pairs.size();

    // Push each card into a bin
    for (int i=0; i < size; i++) // i is each set of card sequences
        for (int j=0; j < pairs[i].size(); j++) // j is each card in the card sequence
            if (! bin[pairs[i][j].rank()].hasCard(pairs[i][j]))
                bin[pairs[i][j].rank()].addCard(pairs[i][j]);

    // Add each new combined sequences to pairs
    pairs.clear();
    for (int i=0; i < Card::NUMBER_OF_RANKS; i++)
        if (! bin[i].isEmpty())
            pairs.push_back(bin[i]);
}

vector<CardSequence> CribbageRules::createPairTypes(const vector<CardSequence> &pairs) const
{
    vector<CardSequence> specialPairs = pairs;

    createPairTypes(specialPairs);
    return specialPairs;
}

CribbageRules::ScorePairs CribbageRules::pairType(const CardSequence &pair) const
{
    switch(pair.size())
    {
    case 2:
        return PAIR;
    case 3:
        return PAIR_THREE;
    case 4:
        return PAIR_FOUR;
    default:
        return PAIR_ERR;
    }
}

void CribbageRules::createRunTypes(vector<CardSequence> &runs) const
{
    CardSequence bin[Card::NUMBER_OF_RANKS];
    CardSequence run;
    int size=runs.size();

    if (size < 3) // Can't be any type of run
        return;
    // Push each card into a bin
    for (int i=0; i < size; i++) // i is each set of card sequences
        for (int j=0; j < runs[i].size(); j++) // j is each card in the card sequence
            if (! bin[runs[i][j].rank()].hasCard(runs[i][j]))
                bin[runs[i][j].rank()].addCard(runs[i][j]);

    // Add each new combined sequences to runs
    runs.clear();
    for (int i=0; i < Card::NUMBER_OF_RANKS; i++) // go through each bin
        for (int j=0; j < bin[i].size(); j++) // in the bin go through each card
            run.addCard(bin[i][j]);
    runs.push_back(run);
}

vector<CardSequence> CribbageRules::createRunTypes(const vector<CardSequence> &runs) const
{
    vector<CardSequence> specialRuns = runs;

    createRunTypes(specialRuns);
    return specialRuns;
}

CribbageRules::ScoreRuns CribbageRules::runType(const vector<CardSequence> &runs) const
{
    switch(runs.size())
    {
    case 1:
        return RUN;
    case 2:
        if (runs.back().size() == 3)
            return RUN_DOUBLE;
        else
            return RUN_DOUBLE_FOUR;
    case 3:
        return RUN_TRIPLE;
    case 4:
        return RUN_QUADRUPLE;
    default:
        return RUN_ERR;
    }
}

bool CribbageRules::isGameOver(const PlayerList &players) const
{
    bool gameOver=false;

    for (int i=0; i < players.size(); i++)
        if (players[i].gamesWon() == NUMBER_OF_STAKES)
            gameOver=true;
    return gameOver;
}

bool CribbageRules::isRoundOver(const PlayerList &players) const
{
    bool hasACard=true;

    for (int i=0; i < players.size(); i++)
        if (players[i].score() >= WINNING_ROUND_SCORE)
            return true;
        else if (! players[i].hand().isEmpty())
            hasACard=false;
    return hasACard;
}

bool CribbageRules::isPhaseOver(const PlayerList &, const CardSequence &) const
{
    throw KardsGTError("CribbageRules", "isPhaseOver", "This is not implemented!");
    return false;
}

int CribbageRules::rankValue(const Card &) const
{
    throw KardsGTError("CribbageRules", "rankValue", "This is not implemented!");
    return Card::RANK_ERR;
}
