/***************************************************************************
*   Copyright (C) 2006 by John Schneiderman                               *
*   JohnMS@member.fsf.org                                                 *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
***************************************************************************/
#include "abigailcribbageai.h"
#include "cardproperties.h"
#include "../../games/cribbage/cribbagerules.h"
#include "kardsgterror.h"

#include <vector>
using std::vector;

AbigailCribbageAI::AbigailCribbageAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand): basicStrategies(rules)
{
    m_playSequence = playSequence;
    m_hand = hand;
    m_pRules = &rules;
}

AbigailCribbageAI::~AbigailCribbageAI()
{}

Card AbigailCribbageAI::selectCard() const
{
    int score = 0;
    CardSequence test, runs;

    // If there is nothing on the board, no need to check special points
    if (! m_playSequence.isEmpty())
    {
        // Calculate the current score
        for (int index = 0, size = m_playSequence.size(); index < size; ++index)
            score += m_pRules->cardValue(m_playSequence[index]);

        // See if we can make 15
        for (int index = 0, size = m_hand.size(); index < size; ++index)
            if ((score + m_pRules->cardValue(m_hand[index])) == 15)
                if (m_pRules->isLegalPlay(m_playSequence, m_hand[index]))
                    return m_hand[index];

        // See if we can make a pair
        for (int index = 0, size = m_hand.size(); index < size; ++index)
            if (m_hand[index].rank() == m_playSequence.back().rank())
                if (m_pRules->isLegalPlay(m_playSequence, m_hand[index]))
                    return m_hand[index];

        // See if we can get a run
        if (m_playSequence.size() > 1)
            for (int i=0; i < m_hand.size(); ++i)
            {
                test.clear();
                test.addCard(m_playSequence.back());
                test.addCard(m_playSequence[m_playSequence.size() - 2]);
                test.addCard(m_hand[i]);
                CardProperties testRun(test);
                if (testRun.runs().size() > 0)
                {
                    runs=testRun.runs().back();
                    if (runs.size() > 0)
                        if (m_pRules->isLegalPlay(m_playSequence, m_hand[i]))
                            return m_hand[i];
                }
            }
    }

    // Player our highest legal card
    int highestCardIndex = 0;
    for (int index = 1, size = m_hand.size(); index < size; ++index)
        if (! m_pRules->isLegalPlay(m_playSequence, m_hand[highestCardIndex]))
            highestCardIndex = index;
        else if (m_pRules->isLegalPlay(m_playSequence, m_hand[index]) && (m_hand[index] > m_hand[highestCardIndex]))
            highestCardIndex = index;
    return m_hand[highestCardIndex];
}

CardSequence AbigailCribbageAI::myCrib() const
{
    vector<int> scores;
    vector<CardSequence> scoreCardsRemoved;
    vector<CardSequence> testHands;
    CardProperties testProp(m_hand);
    int highScoreIndex = 0;

    // Calcualte the different scores when we remove each two-card combinations.
    testHands = testProp.combinations();
    for (int index = 0, endCheck = testHands.size(); index < endCheck; ++index)
    {
        // See if the combinations is four cards
        if (testHands[index].size() != 4)
            continue;

        // Keep track of the cards we're testing
        scoreCardsRemoved.push_back(CardSequence());
        for (int handIndex = 0, size = m_hand.size(); handIndex < size; ++handIndex)
            if (! testHands[index].hasCard(m_hand[handIndex]))
                scoreCardsRemoved.back().addCard(m_hand[handIndex]);

        // Calculate and store the score for that combination
        scores.push_back(calculateScore(testHands[index]));
    }

    // Find the lowest score
    for (int index = 1, size = scores.size(); index < size; ++ index)
        if (scores[index] > scores[highScoreIndex])
            highScoreIndex = index;

    return scoreCardsRemoved[highScoreIndex];
}

CardSequence AbigailCribbageAI::opponentsCrib() const
{
    CardSequence testHand = m_hand; // Remove cards that are worth something
    CardProperties handProp(m_hand);
    CardSequence removeCards;

    removeCards = handProp.ranks(Card::FIVE); // Don't want to give 5s
    removeCards += handProp.ranks(Card::SEVEN); // Don't want to give 7s
    removeCards += handProp.ranks(Card::EIGHT); // Dont' want to give 8s
    if (handProp.runs().size() > 0) // Don't want to give runs away
        removeCards += handProp.runs().back();
    testHand -= removeCards;

    if (testHand.size() == 2) // We have two worthless cards
        return testHand;
    else if (testHand.size() == 1) // We have one worthless card, and need another one
    {
        Card card;

        card = basicStrategies.randomCardsWithNoLegalChecks(1, m_hand).front();
        while (card == testHand.front())
            card = basicStrategies.randomCardsWithNoLegalChecks(1, m_hand).front();
        testHand.addCard(card);
        return testHand;
    }
    else if (testHand.isEmpty()) // All of our cards are woth something, just select two.
        return basicStrategies.randomCardsWithNoLegalChecks(2, m_hand);
    else // Several cards are worthless, just select two.
        return basicStrategies.randomCardsWithNoLegalChecks(2, testHand);
}

int AbigailCribbageAI::calculateScore(const CardSequence &hand) const
{
    CardProperties handProp(hand);
    const CribbageRules *rules = static_cast<const CribbageRules *>(m_pRules);
    vector<CardSequence> pairs, fifteens;
    CribbageRules::ScoreRuns runScore;
    int finalScore = 0;

    // Calculate 15's
    fifteens = handProp.summations(15, *m_pRules);
    for (int i=0; i < static_cast<int>(fifteens.size()); ++i)
        finalScore += 2;

    // Score our pairs
    pairs = rules->createPairTypes(handProp.pairs());
    for (int i=0; i < static_cast<int>(pairs.size()); ++i)
        if (rules->pairType(pairs[i]) != CribbageRules::PAIR_ERR)
            finalScore += rules->pairType(pairs[i]);

    // Score our runs
    runScore = rules->runType(rules->createRunTypes(handProp.runs()));
    if (runScore != CribbageRules::RUN_ERR)
        if (runScore == CribbageRules::RUN)
            finalScore += rules->createRunTypes(handProp.runs()).back().size();
        else
            finalScore += runScore;

    return finalScore;
}
