/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "euchre.h"
#include "euchreinterface.h"
#include "kardsgterror.h"
#include "kardplaysequence.h"
#include "euchreplayer.h"

#include <qstring.h>
#include <qfile.h>

Euchre::Euchre(EuchreInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_deck(), m_playOrder(), m_tricksWon()
{
    m_pInterface = gameInterface;
    m_partnerIndexes[Euchre::PLAYER_ONE_INDEX] = Euchre::PLAYER_THREE_INDEX;
    m_partnerIndexes[Euchre::PLAYER_TWO_INDEX] = Euchre::PLAYER_FOUR_INDEX;
    m_partnerIndexes[Euchre::PLAYER_THREE_INDEX] = Euchre::PLAYER_ONE_INDEX;
    m_partnerIndexes[Euchre::PLAYER_FOUR_INDEX] = Euchre::PLAYER_TWO_INDEX;
    m_deck.removeCards(Card::TWO);
    m_deck.removeCards(Card::THREE);
    m_deck.removeCards(Card::FOUR);
    m_deck.removeCards(Card::FIVE);
    m_deck.removeCards(Card::SIX);
    m_trumpSuit = Card::SUIT_ERR;
    m_turnUpAccepted = true;
    m_makerIndex = NON_PLAYER;
    m_isPlayingAlone = false;
    m_isDefendingAlone = false;
    m_rules.setIsAlone(false);
    m_playerPlayedHighestCard = Euchre::NON_PLAYER;
    teamNSscore = 0, teamEWscore = 0;
    for (int index=0; index < 4; ++index)
        m_tricksWon.push_back(0);
}

Euchre::~Euchre()
{}

void Euchre::startPlay()
{
    time_t seconds;
    int rnd;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("Euchre", "startPlay", "Wrong number of players were added to the game.");
    m_players.swap(Euchre::PLAYER_THREE_INDEX, Euchre::PLAYER_FOUR_INDEX); // Because the array structure is intially different than the visual display.
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    rnd=(rand() % 100) + 1;
    if ((rnd > 75) && (rnd <= 100))
        m_dealerIndex=3;
    else if ((rnd > 50) && (rnd <= 75))
        m_dealerIndex=2;
    else if ((rnd > 25) && (rnd <= 50))
        m_dealerIndex=1;
    else
        m_dealerIndex=0;
    deal();
}

bool Euchre::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(IO_WriteOnly | IO_Translate))
        return false;
    out.setEncoding(QTextStream::UnicodeUTF8);
    m_players.swap(Euchre::PLAYER_FOUR_INDEX, Euchre::PLAYER_THREE_INDEX); // Because the human player must be last.

    out << "Game: " << "euchre" << endl;
    // Begin GameBase Save
    out << "Players: " << endl << m_players << endl;
    out << "RoundPoints: " << m_roundPoints << endl;
    out << "DealerIndex: " << m_dealerIndex << endl;
    out << "PlaySequence: " << m_playSequence << endl;
    // End GameBase Save
    out << "Deck: " << m_deck << endl;
    out << "PlayOrderSize: " << static_cast<int>(m_playOrder.size()) << endl;
    out << "PlayOrderIndexes: ";
    for (int index=0, size = m_playOrder.size(); index < size; ++index)
        out << m_playOrder[index] << " ";
    out << endl;
    out << "TricksWon: ";
    for (int index=0, size = m_tricksWon.size(); index < size; ++index)
        out << m_tricksWon[index] << " ";
    out << endl;
    out << "HighestCardPlayerIndex: " << m_playerPlayedHighestCard << endl;
    out << "TurnUpAccepted: " << m_turnUpAccepted << endl;
    out << "TrumpSuit: " << m_trumpSuit << endl;
    out << "MakerIndex: " << m_makerIndex << endl;
    out << "IsPlayingAlone: " << m_isPlayingAlone << endl;
    out << "IsDefendingAlone: " << m_isDefendingAlone << endl;
    out << "TeamNS_Score: " << teamNSscore << endl;
    out << "TeamEW_Score: " << teamEWscore << endl;
    m_players.swap(Euchre::PLAYER_THREE_INDEX, Euchre::PLAYER_FOUR_INDEX); // Revert so we can continue the game.
    if (file.status() != IO_Ok)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool Euchre::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QString label;
    int value, size;

    if (! file.open(IO_ReadOnly | IO_Translate))
    {
        file.close();
        return false;
    }
    in.setEncoding(QTextStream::UnicodeUTF8);
    in >> label >> label;
    if (label != "euchre")
        return false;
    // Begin GameBase Load
    in >> label >> m_players;
    m_players.swap(Euchre::PLAYER_THREE_INDEX, Euchre::PLAYER_FOUR_INDEX); // Revert so we can continue the game.
    in >> label >> m_roundPoints;
    in >> label >> m_dealerIndex;
    in >> label >> m_playSequence;
    // End GameBase Load
    in >> label >> m_deck;
    m_playOrder.clear();
    in >> label >> size;
    in >> label;
    for (int index = 0; index < size; ++index)
    {
        in >> value;
        m_playOrder.push_back(value);
    }
    m_tricksWon.clear();
    in >> label;
    for (int index=0; index < 4; ++index) // Euchre always has 4 players
    {
        m_tricksWon.push_back(-1);
        in >> m_tricksWon[index];
    }
    in >> label >> value;
    m_playerPlayedHighestCard=static_cast<PlayerIndexes>(value);
    in >> label >> value;
    m_turnUpAccepted = static_cast<bool>(value);
    in >> label >> value;
    m_trumpSuit = static_cast<Card::Suit>(value);
    m_rules.setTrumpSuit(m_trumpSuit);
    in >> label >> value;
    m_makerIndex = static_cast<Euchre::PlayerIndexes>(value);
    in >> label >> value;
    m_isPlayingAlone = static_cast<bool>(value);
    in >> label >> value;
    m_isDefendingAlone = static_cast<bool>(value);
    if (m_isPlayingAlone || m_isDefendingAlone)
        m_rules.setIsAlone(true);
    else
        m_rules.setIsAlone(false);
    in >> label >> teamNSscore;
    in >> label >> teamEWscore;
    file.close();
    m_pInterface->updateTable();
    return true;
}

int Euchre::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int Euchre::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

void Euchre::addOpponent(ComputerPlayer &opponent)
{
    static_cast<EuchrePlayer *>(& opponent)->setRules(m_rules);
    GameBase::addOpponent(opponent);
}

void Euchre::cardPlayed(const Card &card)
{
    // See if he can play the card
    if (m_players.currentPlayer().hand().hasCard(card))
    {
        if (m_rules.isLegalPlay(m_playSequence, card, m_players.currentPlayer()))
        {
            m_playSequence.addCard(card);
            m_players.currentPlayer().hand().removeCard(card);
            m_players[m_partnerIndexes[m_players.currentPlayerIndex()]].gameInformation().add("Partner Played", card.toString());
            m_playOrder.push_back(m_players.currentPlayerIndex());
            m_pInterface->updateTable();
            GameStatusCodes status = static_cast<GameStatusCodes>(handleGameStatus());
            if ((status != Euchre::ROUND_ENDED) && (status != Euchre::GAME_OVER))
                setupNextPlayer();
        }
    }
    else
        throw KardsGTError(QString("Euchre"), QString("cardPlayed"), QString("Failed to find card %1!").arg(card.toString()));
}

void Euchre::deal()
{
    m_pInterface->discardPile->clear(); // We have a delay setup, and so we need to clear the play sequence here.
    nextDealer();
    m_deck.shuffle();
    for (int numDeals = 0; numDeals < 2; ++numDeals)
        for (int index = 0, size = m_players.size(); index < size; ++index)
            for (int cardsToAdd = m_rules.numberOfCardsToDeal(m_players.nextPlayer().hand().size()); cardsToAdd > 0; --cardsToAdd)
                m_players.currentPlayer().hand().addCard(m_deck.dealCard());
    for (int index=0; index < m_players.size(); ++index)
        m_players[index].hand().sortBySuit(); // Give a usefull arrangement.
    m_pInterface->updateTable(); // So the human player can see who is where.
    setupNextPlayer();
    // If trump is not named a new deal is made
    if (! handleMaking())
    {
        reset();
        for (int index=0; index < m_players.size(); ++index)
            m_players[index].hand().clear();
        deal();
    }
    m_pInterface->updateTable();
}

void Euchre::setupNextPlayer()
{
    if (! m_rules.isGameOver(m_players)) // See if the game is over before we start someones turn.
    {
        m_players.currentPlayer().setTurn(false);
        if (m_rules.isPhaseOver(m_players, m_playSequence))
        {
            m_players.setCurrentPlayer(m_players[m_playerPlayedHighestCard]);
            // We clear the play sequence, play order and alone rules tracker here because they are needed to determine if a phase is over.
            m_playSequence.clear();
            m_playOrder.clear();
        }
        else
            m_players.nextNonEmptyPlayer();
        m_players.currentPlayer().setTurn(true);
    }
    else
        for (int index=0; index < m_players.size(); ++index)
            m_players[index].setTurn(false);
}

int Euchre::handleGameStatus()
{
    if (m_rules.isPhaseOver(m_players, m_playSequence))
    {
        handlePhaseEnd();
        if (m_rules.isRoundOver(m_players))
        {
            handleRoundEnd();
            if (m_rules.isGameOver(m_players))
            {
                handleGameOver();
                return Euchre::GAME_OVER;
            }
            else
                return Euchre::ROUND_ENDED;
        }
        else
            return PHASE_ENDED;
    }
    return NO_CHANGE;
}

bool Euchre::handleMaking()
{
    Card::Suit trump;

    m_players.currentPlayer().setTurn(false); // Turn off the turn while we're handeling the making stage of the game.
    trump = makingWithTurnUp();
    if (trump == Card::SUIT_ERR) // All turned it down
    {
        m_turnUpAccepted = false;
        m_pInterface->updateTable();
        trump = makingAllPassed();
        if (trump == Card::SUIT_ERR) // Everyone passed
        {
            m_players.setCurrentPlayer(m_players.dealer());
            return false;
        }
    }
    else // The turn up was accepted that means the dealer can choose to take it.
        handleDealerWantsTurnUp();
    // Trump has been determined
    m_trumpSuit = trump;
    m_rules.setTrumpSuit(m_trumpSuit);
    m_makerIndex = static_cast<PlayerIndexes>(m_players.currentPlayerIndex());
    m_pInterface->displayMaker();
    handlePlayAlone();
    // Against a lone maker the player to his left plays next, else use the dealer
    if (m_isPlayingAlone)
    {
        m_players.setCurrentPlayer(m_players[m_makerIndex]);
        m_players.nextPlayer(); // Maker's left
        m_players.currentPlayer().setTurn(true);
    }
    else
    {
        m_players.setCurrentPlayer(m_players.dealer());
        m_players.nextPlayer(); // Dealer's left
        m_players.currentPlayer().setTurn(true);
    }
    m_pInterface->updateTable();
    return true;
}

Card::Suit Euchre::makingWithTurnUp()
{
    EuchrePlayer *player = 0;

    for (int index = 0, size = m_players.size(); index < size; ++index)
    {
        player = dynamic_cast<EuchrePlayer *>(& m_players.currentPlayer());

        if (player == 0) // Handle human player
        {
            if (m_pInterface->makeTurnUp())
                return m_deck.topCard().suit();
        }
        else
            if (player->makeTurnUp(m_deck.topCard().suit()))
                return m_deck.topCard().suit();
        m_players.nextPlayer();
    }
    return Card::SUIT_ERR;
}

Card::Suit Euchre::makingAllPassed()
{
    EuchrePlayer *player = 0;
    Card::Suit trump;

    for (int index = 0, size = m_players.size(); index < size; ++index)
    {
        player = dynamic_cast<EuchrePlayer *>(& m_players.currentPlayer());

        if (player == 0) // Handle human player
        {
            trump = m_pInterface->selectTrump(m_deck.topCard().suit());
            if (trump != Card::SUIT_ERR)
                return trump;
        }
        else
        {
            trump = player->selectTrump(m_deck.topCard().suit());
            if (trump != Card::SUIT_ERR)
                return trump;
        }
        m_players.nextPlayer();
    }
    return Card::SUIT_ERR;
}

void Euchre::handleDealerWantsTurnUp()
{
    EuchrePlayer *player = dynamic_cast<EuchrePlayer *>(& m_players.dealer());

    if (player == 0) // Handle human player
    {
        if (m_pInterface->dealerWantsTurnUp())
        {
            Card discard;

            discard = m_pInterface->discard();
            m_players.dealer().hand().removeCard(discard);
            m_players.dealer().hand().addCard(m_deck.dealCard());
            m_turnUpAccepted = false; // Even though the turn-up was accepted it is now located in the dealer's hand.
        }
    }
    else
        if (player->dealerWantsTurnUp(m_deck.topCard()))
        {
            Card discard;

            discard = player->discard();
            player->hand().removeCard(discard);
            player->hand().addCard(m_deck.dealCard());
            m_turnUpAccepted = false; // Even though the turn-up was accepted it is now located in the dealer's hand.
        }
}

void Euchre::handlePlayAlone()
{
    EuchrePlayer *player = dynamic_cast<EuchrePlayer *>(& m_players.currentPlayer());

    if (player == 0) // Handle human player
    {
        if (m_pInterface->playAlone())
        {
            m_isPlayingAlone = true;
            lonePlayerSetup();
        }
    }
    else
        if (player->playAlone())
        {
            m_isPlayingAlone = true;
            lonePlayerSetup();
        }
}

void Euchre::handleDefendAlone()
{
    EuchrePlayer *player = dynamic_cast<EuchrePlayer *>(& m_players.currentPlayer());

    if (player == 0) // Handle human player
    {
        if (m_pInterface->defendAlone())
        {
            m_isDefendingAlone = true;
            lonePlayerSetup();
        }
    }
    else
        if (player->defendAlone())
        {
            m_isDefendingAlone = true;
            lonePlayerSetup();
        }
}

void Euchre::lonePlayerSetup()
{
    m_rules.setIsAlone(true);
    m_players[m_partnerIndexes[m_players.currentPlayerIndex()]].hand().clear();
    m_pInterface->updateTable();
}

void Euchre::handlePhaseEnd()
{
    m_playerPlayedHighestCard = playerPlayedHighestCard();
    m_tricksWon[m_playerPlayedHighestCard]++;
    for (int index = 0; index < m_players.size(); ++index) // Clear the partner played information
        m_players[index].gameInformation().add("Partner Played", "--");
}

Euchre::PlayerIndexes Euchre::playerPlayedHighestCard()
{
    int highCardFound=0;
    Card::Suit leadSuit = m_playSequence.front().suit();

    for (int cardIndex=0; cardIndex < m_playSequence.size(); ++cardIndex)
        if ((m_playSequence[cardIndex].suit() == leadSuit) || (m_playSequence[cardIndex].suit() == m_trumpSuit) || m_rules.isLeftBower(m_playSequence[cardIndex])) // It has to be of the lead suit, trump, or the left bower.
            if (m_rules.rankValue(m_playSequence[cardIndex]) > m_rules.rankValue(m_playSequence[highCardFound]))
                highCardFound = cardIndex;
    return static_cast<Euchre::PlayerIndexes>(m_playOrder[highCardFound]);
}

void Euchre::handleRoundEnd()
{
    for (int index=0; index < m_players.size(); ++index)
        m_players[index].setTurn(false);
    m_pInterface->updateTable(); // So that the table doesn't look like the last player hasn't played his card.
    handleRoundSummary(handleScoring()); // Creates the final round summay information and show it to the user.
    reset();
    if (! m_rules.isGameOver(m_players))
    {
        // Set the opponents score information
        for (int index=0; index < (m_players.size() - 1); ++index)
            m_players[index].gameInformation().add("Opponents Score", m_players[index + 1].score());
        m_players[m_players.size() - 1].gameInformation().add("Opponents Score", m_players[0].score());
        deal();
    }
}

int Euchre::handleScoring()
{
    int score = 0;

    teamNSscore = m_tricksWon[PLAYER_ONE_INDEX] + m_tricksWon[PLAYER_THREE_INDEX];
    teamEWscore = m_tricksWon[PLAYER_TWO_INDEX] + m_tricksWon[PLAYER_FOUR_INDEX];

    if (teamNSscore >= m_rules.pointsToWinRound())
    {
        score = calculateScore(PLAYER_ONE_INDEX, PLAYER_THREE_INDEX, teamNSscore);
        m_players[PLAYER_ONE_INDEX].addPoint(score);
        m_players[PLAYER_THREE_INDEX].addPoint(score);
    }
    else
    {
        score = calculateScore(PLAYER_TWO_INDEX, PLAYER_FOUR_INDEX, teamEWscore);
        m_players[PLAYER_TWO_INDEX].addPoint(score);
        m_players[PLAYER_FOUR_INDEX].addPoint(score);
    }
    return score;
}

int Euchre::calculateScore(PlayerIndexes player1, PlayerIndexes player2, int teamScore) const
{
    int finalScore = 0;

    // Is the team the making side
    if ((m_makerIndex == player1) || (m_makerIndex == player2))
        if (m_isPlayingAlone) // Are both team members playing?
            if (m_rules.isMarch(teamScore)) // Is the score a march?
                finalScore = m_rules.makerAloneMarchPoint();
            else
                finalScore = m_rules.makerAlonePoint();
        else // not playing alone
            if (m_rules.isMarch(teamScore)) // Is the score a march?
                finalScore = m_rules.makerMarchPoint();
            else
                finalScore = m_rules.makerPoint();
    else
        if (m_isDefendingAlone) // Are both team members playing?
            if (m_rules.isMarch(teamScore)) // Is the score a march?
                finalScore = m_rules.defenderAloneMarchPoint();
            else
                finalScore = m_rules.defenderAlonePoint();
        else
            finalScore = m_rules.defenderPoint();
    return finalScore;
}

void Euchre::handleRoundSummary(int score)
{
    QString message;

    // Create message for North-South Team
    if (teamNSscore >= m_rules.pointsToWinRound())
    {
        if (m_rules.isMarch(teamNSscore))
            message = QString(m_players[PLAYER_ONE_INDEX].name() + " & " + m_players[PLAYER_THREE_INDEX].name() + " marched, ");
        else
            message = QString(m_players[PLAYER_ONE_INDEX].name() + " & " + m_players[PLAYER_THREE_INDEX].name() + " made %1 tricks, ").arg(teamNSscore);
        message += QString("earning %1 points.\n").arg(score);
    }
    else // lost round
        if ((m_makerIndex == PLAYER_ONE_INDEX) || (m_makerIndex == PLAYER_THREE_INDEX))
            message = QString(m_players[PLAYER_ONE_INDEX].name() + " & " + m_players[PLAYER_THREE_INDEX].name() + " were euchred!\n");
        else
            message = QString(m_players[PLAYER_ONE_INDEX].name() + " & " + m_players[PLAYER_THREE_INDEX].name() + " only made %1 tricks.\n").arg(teamNSscore);
    // Create message for East-West Team
    if (teamEWscore >= m_rules.pointsToWinRound()) // won round
    {
        if (m_rules.isMarch(teamEWscore)) // marched
            message += QString(m_players[PLAYER_TWO_INDEX].name() + " & " + m_players[PLAYER_FOUR_INDEX].name() + " marched, ");
        else
            message += QString(m_players[PLAYER_TWO_INDEX].name() + " & " + m_players[PLAYER_FOUR_INDEX].name() + " made %1 tricks, ").arg(teamEWscore);
        message += QString("earning %1 points.\n").arg(score);
    }
    else // lost round
        if ((m_makerIndex == PLAYER_TWO_INDEX) || (m_makerIndex == PLAYER_FOUR_INDEX)) // lost as makers
            message += QString(m_players[PLAYER_TWO_INDEX].name() + " & " + m_players[PLAYER_FOUR_INDEX].name() + " were euchred!\n");
        else
            message += QString(m_players[PLAYER_TWO_INDEX].name() + " & " + m_players[PLAYER_FOUR_INDEX].name() + " only made %1 tricks.\n").arg(teamEWscore);
    m_pInterface->promptMessage("Round Summary", message);
}

void Euchre::reset()
{
    // Clear any game information saved
    for (int index =0; index < m_players.size(); ++index)
    {
        m_players[index].gameInformation().clear();
        m_tricksWon[index] = 0;
    }
    m_isPlayingAlone = m_isDefendingAlone = false;
    m_turnUpAccepted = true;
    m_rules.setTrumpSuit(Card::SUIT_ERR);
    m_trumpSuit = Card::SUIT_ERR;
    teamNSscore = 0, teamEWscore = 0;
    m_playerPlayedHighestCard = NON_PLAYER;
    m_rules.setIsAlone(false);
    m_playSequence.clear();
    m_playOrder.clear();
}

void Euchre::handleGameOver()
{
    m_pInterface->gameWon();
}
