/***************************************************************************
 *   Copyright (C) 2007 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef EUCHRE_H
#define EUCHRE_H

#include "gamebase.h"
#include "euchrerules.h"
#include "carddeck.h"

class EuchreInterface;

/**
 * This controls and keeps track of the state of the game euchre.
 *
 * @author John Schneiderman
 */
class Euchre: public GameBase
{
public:
    /**
     * The constructor for the euchre.
     * @param gameInterface is the interface for the game.
     * @param profileDatabase is the user profile database.
     */
    Euchre(EuchreInterface *gameInterface, UserProfileDatabase &profileDatabase);
    /**
     * The destructor.
     */
    ~Euchre();
    /**
     * This starts the game of euchre.
     */
    virtual void startPlay();
    /**
     * This saves the game of euchre.
     * @param filename is the name of the file to save the game to.
     * @return true if we successfully save the game, false elsewise.
     */
    virtual bool save(const QString &filename);
    /**
     * This loads the game of euchre.
     * @param filename is the name of the file to load.
     * @return true if we load the game successfully, false elsewise.
     */
    virtual bool load(const QString &filename);
    /**
     * This gives the minimum number of players needed to play the game of euchre.
     * @return the minimum number of players.
     */
    virtual int minimumPlayers() const;
    /**
     * This gives the maximum number of players allowed to play the game of euchre.
     * @return the maximum number of players.
     */
    virtual int maximumPlayers() const;
    /**
     * Adds the opponent to play against the player, and sets their rule pointer.
     * @param opponent is the player to play against our local player.
     */
    virtual void addOpponent(ComputerPlayer &opponent);

protected:
    /**
     * These are the indexes for each player.
     * @param PLAYER_ONE_INDEX is the index for player one.
     * @param PLAYER_TWO_INDEX is the index for player two.
     * @param PLAYER_THREE_INDEX is the index for player three.
     * @param PLAYER_FOUR_INDEX is the index for player four.
     * @param NON_PLAYER is the index for an invalid player.
     */
    enum PlayerIndexes { PLAYER_ONE_INDEX=0, PLAYER_TWO_INDEX=1, PLAYER_THREE_INDEX=2, PLAYER_FOUR_INDEX=3, NON_PLAYER=-1 };

    /// @param m_rules is the rules to the game of euchre.
    EuchreRules m_rules;
    /// @param m_deck is the deck of cards used by the players.
    CardDeck m_deck;
    /// @param m_playOrder is the player index order for the cards that each player played.
    vector<int> m_playOrder;
    /// @param playerPlayedHighestCard is the index to the player who played the highest card.
    PlayerIndexes m_playerPlayedHighestCard;
    /// @param m_turnUpAccepted indicates if the turn up card was accepted by a player. If it is not, then the turn-up card should not be displayed.
    bool m_turnUpAccepted;
    /// @param m_trumpSuit is the current trump suit.
    Card::Suit m_trumpSuit;
    /// @param m_tricksWon holds the number of tricks won by each player.
    vector<int> m_tricksWon;
    /// @param m_makerIndex iis the index of the maker.
    PlayerIndexes m_makerIndex;

    /**
     * Handles a card being played.
     * @param card is the card the player wants to play.
     */
    virtual void cardPlayed(const Card &card);
    /**
     * Deals out the cards to each player.
     * @note We deal the cards in a 3-2 configuration instead of the 2-3.
     */
    virtual void deal();
    /**
     * Disables the current player, and enables the next player's turn.
     */
    virtual void setupNextPlayer();
    /**
     * Determines and handles the various stages in game play.
     * @return the last major GameStatusCode value after handling the changes.
     */
    virtual int handleGameStatus();

private:
    /**
     * These are the game ending states for Spades.
     * @param NO_CHANGE indicates there has been no change in the current state.
     * @param PHASE_ENDED indicates that the phase has ended.
     * @param ROUND_ENDED indicates that the round has ended.
     * @param GAME_OVER indicates that the game has ended.
     */
    enum GameStatusCodes
    {
        NO_CHANGE = 0,
        PHASE_ENDED = 1,
        ROUND_ENDED = 2,
        GAME_OVER = 3
    };

    /// @param m_pInterface is the interface to our graphical interface.
    EuchreInterface *m_pInterface;
    /// @param m_isPlayingAlone is true if the maker is playing alone, false elsewise.
    bool m_isPlayingAlone;
    /// @param m_isDefendingAlone is true if the a defender of the maker is defending alone, false elsewise.
    bool m_isDefendingAlone;
    /// @param m_partnerIndexes is the indexes of each partner.
    PlayerIndexes m_partnerIndexes[4];
    /// @param teamNSscore is the team score for the North-South players.
    int teamNSscore;
    /// @param teamEWscore is the team score for the East-West players.
    int teamEWscore;

    /**
     * Handles the making phase of the game.
     * @return true if trump is determined, false if it has not.
     */
    bool handleMaking();
    /**
     * Handles the players decisions to accept the turn-up card.
     * @return the turn-up card's suit if a player wanted it for trump, SUIT_ERR elsewise.
     */
    Card::Suit makingWithTurnUp();
    /**
     * Handles the players decisions to declare a trump suit, if all pass for the turn-up card.
     * @return the turn-up card's suit if a player wanted it for trump, SUIT_ERR elsewise.
     */
    Card::Suit makingAllPassed();
    /**
     * Handles if the dealer wants the turn-up card, when a player accepts the turn-up card suit for trump.
     */
    void handleDealerWantsTurnUp();
    /**
     * Handles if the maker wishes to play alone.
     */
    void handlePlayAlone();
    /**
     * Handles if the opponent of the maker wishes to play alone.
     */
    void handleDefendAlone();
    /**
     * Setups the game if the maker wants to play alone.
     */
    void lonePlayerSetup();
    /**
     * Handles the end of the phase.
     */
    void handlePhaseEnd();
    /**
     * Searches through the playing sequence and finds the highest card or highest trump that was played.
     * @return the index of the player who played the highest card or trump.
     */
    PlayerIndexes playerPlayedHighestCard();
    /**
     * Handles the end of the round.
     */
    void handleRoundEnd();
    /**
     * Handles the scoring for each partnership.
     * @return the score of the winning team.
     */
    int handleScoring();
    /**
     * Calcualtes the score.
     * @param player1 is the first player on the team.
     * @param player2 is the second player on the team.
     * @param teamScore is the combined scores of each player.
     * @return the points made in the game.
     */
    int calculateScore(PlayerIndexes player1, PlayerIndexes player2, int teamScore) const;
    /**
     * Handles the creation of the round summary.
     * @param score is the score of the winning team.
     */
    void handleRoundSummary(int score);
    /**
     * Resets the game back to it's beginning round state.
     */
    void reset();
    /**
     * This handles all the tasks for the end of the game.
     */
    void handleGameOver();
};
#endif
