/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#include "spades.h"
#include "spadesinterface.h"
#include "kardsgterror.h"
#include "kardplaysequence.h"
#include "spadesplayer.h"
#include "gameinformation.h"

#include <qfile.h>

Spades::Spades(SpadesInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_deck(), m_tricksWon(), m_contractMade()
{
    m_pInterface = gameInterface;
    m_partnerIndexes[Spades::PLAYER_ONE_INDEX] = Spades::PLAYER_THREE_INDEX;
    m_partnerIndexes[Spades::PLAYER_TWO_INDEX] = Spades::PLAYER_FOUR_INDEX;
    m_partnerIndexes[Spades::PLAYER_THREE_INDEX] = Spades::PLAYER_ONE_INDEX;
    m_partnerIndexes[Spades::PLAYER_FOUR_INDEX] = Spades::PLAYER_TWO_INDEX;
    // Create a holder for each of the player's m_playerBids
    for (int index = 0; index < m_rules.minimumNumberOfPlayers(); ++index)
        m_playerBids.push_back(SpadesRules::NON_BID);
    m_playerPlayedHighestCard = Spades::NON_PLAYER;
}

Spades::~Spades()
{}

void Spades::startPlay()
{
    time_t seconds;
    int rnd;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("Spades", "startPlay", "Wrong number of players were added to the game.");
    m_players.swap(Spades::PLAYER_THREE_INDEX, Spades::PLAYER_FOUR_INDEX); // Because the array structure is intially different than the visual display.
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    rnd=(rand() % 100) + 1;
    if ((rnd > 75) && (rnd <= 100))
        m_dealerIndex=3;
    else if ((rnd > 50) && (rnd <= 75))
        m_dealerIndex=2;
    else if ((rnd > 25) && (rnd <= 50))
        m_dealerIndex=1;
    else
        m_dealerIndex=0;
    deal();
}

bool Spades::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(IO_WriteOnly | IO_Translate))
        return false;
    out.setEncoding(QTextStream::UnicodeUTF8);
    m_players.swap(Spades::PLAYER_FOUR_INDEX, Spades::PLAYER_THREE_INDEX); // Because the human player must be last.

    out << "Game: " << "spades" << endl;
    // Begin GameBase Save
    out << "Players: " << endl << m_players << endl;
    out << "RoundPoints: " << m_roundPoints << endl;
    out << "DealerIndex: " << m_dealerIndex << endl;
    out << "PlaySequence: " << m_playSequence << endl;
    // End GameBase Save
    out << "Deck: " << m_deck << endl;
    out << "CardOrderIndexesSize: " << static_cast<int>(m_playerCardOrderIndexes.size()) << endl;
    out << "CardOrderIndexes: ";
    for (int i=0; i < static_cast<int>(m_playerCardOrderIndexes.size()); ++i)
        out << m_playerCardOrderIndexes[i] << " ";
    out << endl;
    out << "TricksWon: ";
    for (int i=0; i < static_cast<int>(m_tricksWon.size()); ++i)
        out << m_tricksWon[i] << " ";
    out << endl;
    out << "HighestCardPlayerIndex: " << m_playerPlayedHighestCard << endl;
    out << "PlayerBids: ";
    for (int i=0; i < static_cast<int>(m_playerBids.size()); ++i)
        out << m_playerBids[i] << " ";
    out << endl;
    out << "SpadesBroken: " << m_rules.isSpadesBroken() << endl;
    m_players.swap(Spades::PLAYER_THREE_INDEX, Spades::PLAYER_FOUR_INDEX); // Revert so we can continue the game.
    if (file.status() != IO_Ok)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool Spades::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QString label;
    int value, size;

    if (! file.open(IO_ReadOnly | IO_Translate))
    {
        file.close();
        return false;
    }
    in.setEncoding(QTextStream::UnicodeUTF8);
    in >> label >> label;
    if (label != "spades")
        return false;
    // Begin GameBase Load
    in >> label >> m_players;
    m_players.swap(Spades::PLAYER_THREE_INDEX, Spades::PLAYER_FOUR_INDEX); // Revert so we can continue the game.
    in >> label >> m_roundPoints;
    in >> label >> m_dealerIndex;
    in >> label >> m_playSequence;
    // End GameBase Load
    in >> label >> m_deck;
    m_playerCardOrderIndexes.clear();
    in >> label >> size;
    in >> label;
    for (int i=0; i < size; ++i)
    {
        in >> value;
        m_playerCardOrderIndexes.push_back(static_cast<PlayerIndexes>(value));
    }
    m_tricksWon.clear();
    in >> label;
    for (int i=0; i < 4; ++i)
    {
        m_tricksWon.push_back(0);
        in >> m_tricksWon[i];
    }
    in >> label >> value;
    m_playerPlayedHighestCard=static_cast<PlayerIndexes>(value);
    m_playerBids.clear();
    in >> label;
    for (int i=0; i < 4; ++i) // Spades always has 4 players
    {
        m_playerBids.push_back(0);
        in >> m_playerBids[i];
    }
    in >> label >> value;
    m_rules.setSpadesBroken(static_cast<bool>(value));
    file.close();
    m_pInterface->updateTable();
    return true;
}

int Spades::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int Spades::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

void Spades::addOpponent(ComputerPlayer &opponent)
{
    static_cast<SpadesPlayer *>(& opponent)->setRules(m_rules);
    GameBase::addOpponent(opponent);
}

void Spades::cardPlayed(const Card &card)
{
    int currentIndex = -1;

    // Find the index of the current player
    for (int index = 0; index < m_players.size(); ++index)
        if (m_players[index] == m_players.currentPlayer())
        {
            currentIndex = index;
            break;
        }
    // See if he can play the card
    if (m_players.currentPlayer().hand().hasCard(card))
    {
        if (m_rules.isLegalPlay(m_playSequence, card, m_players.currentPlayer()))
        {
            m_playSequence.addCard(card);
            m_players.currentPlayer().hand().removeCard(card);
            m_players[m_partnerIndexes[currentIndex]].gameInformation().add("Partner Played", card.toString());
            setupCardOrder();
            m_pInterface->updateTable();
            GameStatusCodes status = static_cast<GameStatusCodes>(handleGameStatus());
            if ((status != Spades::ROUND_ENDED) && (status != Spades::GAME_OVER))
                setupNextPlayer();
        }
    }
    else
        throw KardsGTError(QString("Spades"), QString("cardPlayed"), QString("Failed to find card %1").arg(card.toString()));
}

void Spades::deal()
{
    m_pInterface->playSequence->clear(); // We have a delay setup, and so we need to clear the play sequence here.
    for (int i = 0; i < m_players.size(); ++i)
    {
        m_players[i].hand().clear();
        m_tricksWon.push_back(0);
    }
    nextDealer();
    m_deck.shuffle();
    for (int i = 0; i < m_rules.numberOfCardsToDeal(); ++i)
        for (int j = 0; j < m_players.size(); ++j)
            m_players.nextPlayer().hand().addCard(m_deck.dealCard());
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].hand().sortBySuit(); // Give a usefull arrangement.
    m_pInterface->showCards(false, Spades::PLAYER_THREE_INDEX); // Hide the human players cards
    m_pInterface->updateTable(); // So the human player can see who is where.
    handleBid();
    setupNextPlayer();
    m_pInterface->updateTable();
}

void Spades::setupNextPlayer()
{
    if (! m_rules.isGameOver(m_players)) // See if the game is over before we start someones turn.
    {
        m_players.currentPlayer().setTurn(false);
        if (m_rules.isPhaseOver(m_players, m_playSequence))
        {
            m_players.setCurrentPlayer(m_players[m_playerPlayedHighestCard]);
            m_playSequence.clear(); // Clear the play sequence for the next phase. We clear it here because we need the check if the phase is over.
        }
        else
            m_players.nextNonEmptyPlayer();
        m_players.currentPlayer().setTurn(true);
    }
    else
        for (int i=0; i < m_players.size(); ++i)
            m_players[i].setTurn(false);
}

int Spades::handleGameStatus()
{
    if (m_rules.isPhaseOver(m_players, m_playSequence))
    {
        handelEndOfPhase();
        if (m_rules.isRoundOver(m_players))
        {
            handleEndOfRound();
            if (m_rules.isGameOver(m_players))
            {
                handleGameOver();
                return Spades::GAME_OVER;
            }
            else
                return Spades::ROUND_ENDED;
        }
        else
            return Spades::PHASE_ENDED;
    }
    return Spades::NO_CHANGE;
}

void Spades::setupCardOrder()
{
    if (m_players.currentPlayer() == m_players[Spades::PLAYER_ONE_INDEX])
        m_playerCardOrderIndexes.push_back(Spades::PLAYER_ONE_INDEX);
    else if (m_players.currentPlayer() == m_players[Spades::PLAYER_TWO_INDEX])
        m_playerCardOrderIndexes.push_back(Spades::PLAYER_TWO_INDEX);
    else if (m_players.currentPlayer() == m_players[Spades::PLAYER_THREE_INDEX])
        m_playerCardOrderIndexes.push_back(Spades::PLAYER_THREE_INDEX);
    else if (m_players.currentPlayer() == m_players[Spades::PLAYER_FOUR_INDEX])
        m_playerCardOrderIndexes.push_back(Spades::PLAYER_FOUR_INDEX);
    else
        m_playerCardOrderIndexes.push_back(Spades::NON_PLAYER);
}

void Spades::handleBid()
{
    vector<CardSequence> exchangeCards;
    int numberOfBids = 0;
    int bidderIndex = -1;

    // Create a holder for each of the player
    for (int index = 0; index < m_players.size(); ++index)
    {
        exchangeCards.push_back(CardSequence());
        if (m_players[index].isDealer())
            bidderIndex = index;
    }
    // Let players make their bid, starting with the dealer
    while (numberOfBids < m_players.size())
    {
        SpadesPlayer *spadesPlayer=dynamic_cast<SpadesPlayer *>(& m_players[bidderIndex]);

        // Get the player's bid
        if (spadesPlayer == 0) // Handle human player
        {
            if (m_pInterface->bidDouble())
                m_playerBids[bidderIndex] = SpadesRules::DOUBLE_NIL;
            else // Get bid
            {
                m_pInterface->showCards(true, static_cast<Spades::PlayerIndexes>(bidderIndex)); // Show the human players cards
                m_playerBids[bidderIndex] = m_pInterface->bid();
            }
        }
        else // Handle computer player
        {
            // Offer a chance for double nil
            if (spadesPlayer->bidDoubleNil() == SpadesRules::DOUBLE_NIL)
                m_playerBids[bidderIndex] = SpadesRules::DOUBLE_NIL;
            else // Get bid
                m_playerBids[bidderIndex] = spadesPlayer->bid();
        }
        // Check to see if player bid nil or double nil
        if ((m_playerBids[bidderIndex] == SpadesRules::NIL) || (m_playerBids[bidderIndex] == SpadesRules::DOUBLE_NIL))
            if (spadesPlayer == 0) // Handle human player
                exchangeCards[bidderIndex] = m_pInterface->selectExchangeCards(m_players[bidderIndex], m_players[m_partnerIndexes[bidderIndex]]);
            else
                exchangeCards[bidderIndex] = spadesPlayer->selectExchangeCards();
        // Have the player announce their bid
        m_pInterface->displayBid(static_cast<Spades::PlayerIndexes>(bidderIndex), m_playerBids[bidderIndex]);
        // Store the current bidder's bid so the partner knows.
        m_players[m_partnerIndexes[bidderIndex]].gameInformation().add("Partners Bid", m_playerBids[bidderIndex]);
        // Setup for the next player
        bidderIndex = (bidderIndex + 1) % m_players.size();
        ++numberOfBids;
    }
    // See if there are any cards to exchange, because of a nil or double nil bid
    for (int index = 0; index < m_players.size(); ++index)
        if (! exchangeCards[index].isEmpty()) // Did player at index bid nill or double nil?
        {
            if (exchangeCards[m_partnerIndexes[index]].isEmpty()) // Make sure the partner didn't bid nil or double nil too
                handleExchangePartnerCards(static_cast<PlayerIndexes>(index), exchangeCards[index]);
            // Store that the player has made a nil bid
            m_players[index].gameInformation().add("Made Nil Bid", "true");
        }
}

void Spades::handleExchangePartnerCards(PlayerIndexes playersIndex, const CardSequence &playersCards)
{
    SpadesPlayer *spadesPartnerPlayer=dynamic_cast<SpadesPlayer *>(& m_players[m_partnerIndexes[playersIndex]]);
    CardSequence partnersCards;

    // Get the cards the partner wants to exchange
    if (spadesPartnerPlayer == 0)
        partnersCards = m_pInterface->selectExchangeCards(m_players[m_partnerIndexes[playersIndex]], m_players[playersIndex]);
    else
        partnersCards = spadesPartnerPlayer->selectExchangeCards();
    // Exchange the cards
    for (int i=0; i < SpadesRules::CARDS_TO_EXCHANGE; ++i)
    {
        m_players[playersIndex].hand().removeCard(playersCards[i]); // Remove the players card
        m_players[m_partnerIndexes[playersIndex]].hand().addCard(playersCards[i]); // Add the card to his partner's cards
        m_players[m_partnerIndexes[playersIndex]].hand().removeCard(partnersCards[i]); // Remove the partner's card
        m_players[playersIndex].hand().addCard(partnersCards[i]); // Add the card to the players cards
    }
}

void Spades::handelEndOfPhase()
{
    m_playerPlayedHighestCard=playerPlayedHighestCard();
    m_tricksWon[m_playerPlayedHighestCard]++;
    m_playerCardOrderIndexes.clear(); // We're going to start a new phase, clear our card order tracker.
    for (int index = 0; index < m_players.size(); ++index)
        m_players[index].gameInformation().add("Partner Played", "--");
}

Spades::PlayerIndexes Spades::playerPlayedHighestCard()
{
    int highCardFound=0;
    Card::Suit leadingSuit;
    bool trumpPlayed = false;

    leadingSuit = m_playSequence.front().suit();
    for (int cardIndex=0; cardIndex < m_playSequence.size(); ++cardIndex)
        if ((! trumpPlayed) && (m_playSequence[cardIndex].suit() == leadingSuit))
        {
            if (m_rules.rankValue(m_playSequence[cardIndex]) > m_rules.rankValue(m_playSequence[highCardFound]))
                highCardFound = cardIndex;
        }
        else if (m_playSequence[cardIndex].suit() == Card::SPADES)
        {
            if (m_playSequence[highCardFound].suit() == Card::SPADES)
            {
                if (m_rules.rankValue(m_playSequence[cardIndex]) > m_rules.rankValue(m_playSequence[highCardFound]))
                    highCardFound = cardIndex;
            }
            else
                highCardFound = cardIndex;
            trumpPlayed = true;
            m_rules.setSpadesBroken(true);
        }
    return m_playerCardOrderIndexes[highCardFound];
}

void Spades::handleEndOfRound()
{
    for (int i=0; i < m_players.size(); ++i)
        m_players[i].setTurn(false);
    m_pInterface->updateTable(); // So that the table doesn't look like the last player hasn't played his card.
    handleScoring(); // Take care of the players score's.
    handleRoundSummary(); // Creates the final round summay information and show it to the user.
    m_playerPlayedHighestCard = NON_PLAYER;
    reset();
    if (! m_rules.isGameOver(m_players))
    {
        // Set the opponents score information
        for (int i=0; i < (m_players.size() - 1); ++i)
            m_players[i].gameInformation().add("Opponents Score", m_players[i + 1].score());
        m_players[m_players.size() - 1].gameInformation().add("Opponents Score", m_players[0].score());
        deal();
    }
}

void Spades::handleScoring()
{
    int teamSize = m_players.size() / 2, points = 0;

    for (int index=0; index < teamSize; ++index)
    {
        points = calculateScore(pair<int, int>(index, m_partnerIndexes[index]));
        m_players[index].addPoint(points);
        m_players[m_partnerIndexes[index]].addPoint(points);
        if (points > 0)
            if (index == 0)
                m_contractMade.first = true;
            else
                m_contractMade.second = true;
        else
            if (index == 0)
                m_contractMade.first = false;
            else
                m_contractMade.second = false;
    }
}

int Spades::calculateScore(pair<int, int> teamIndex) const
{
    int contract;
    int tricksMade;
    int score = 0;

    contract = calculateContract(teamIndex);
    tricksMade = calculateTricksWon(teamIndex);

    // Was the contract made
    if (tricksMade != SpadesRules::DOUBLE_NIL)
    {
        if (tricksMade >= contract)
        {
            score += contract * SpadesRules::TRICKS_MADE_CONTRACT;
            score += (tricksMade - contract) * SpadesRules::TRICKS_EXCESS_CONTRACT;
        }
        else
            score -= contract * SpadesRules::TRICKS_MADE_CONTRACT;
    }
    // Did a player bid nil or double nil
    if (! bothPlayersBidNil(teamIndex))
    {
        if ((m_playerBids[teamIndex.first] == SpadesRules::NIL) || (m_playerBids[teamIndex.first] == SpadesRules::DOUBLE_NIL))
        {
            if (m_tricksWon[teamIndex.first] == 0) // Won the nil bid
                if (m_playerBids[teamIndex.first] == SpadesRules::NIL)
                    score += SpadesRules::NIL_SCORE;
                else
                    score += SpadesRules::DOUBLE_NIL_SCORE;
            else // Lost nil bid
                if (m_playerBids[teamIndex.first] == SpadesRules::NIL)
                    score -= SpadesRules::NIL_SCORE;
                else
                    score -= SpadesRules::DOUBLE_NIL_SCORE;
        }
        if ((m_playerBids[teamIndex.second] == SpadesRules::NIL) || (m_playerBids[teamIndex.second] == SpadesRules::DOUBLE_NIL))
        {
            if (m_tricksWon[teamIndex.second] == 0) // Won the nil bid
                if (m_playerBids[teamIndex.second] == SpadesRules::NIL)
                    score += SpadesRules::NIL_SCORE;
                else
                    score += SpadesRules::DOUBLE_NIL_SCORE;
            else // Lost nil bid
                if (m_playerBids[teamIndex.second] == SpadesRules::NIL)
                    score -= SpadesRules::NIL_SCORE;
                else
                    score -= SpadesRules::DOUBLE_NIL_SCORE;
        }
    }
    else if (bothPlayersBidNil(teamIndex)) // Since both bid nil can only score if both make their contracts
        if ((m_tricksWon[teamIndex.first] == 0) && (m_tricksWon[teamIndex.second] == 0))
            score += SpadesRules::PARTNERSHIP_NIL_SCORE;
        else
            score = 0;
    return score;
}

bool Spades::bothPlayersBidNil(pair<int, int> teamIndex) const
{
    if (((m_playerBids[teamIndex.first] == SpadesRules::NIL) || (m_playerBids[teamIndex.first] == SpadesRules::DOUBLE_NIL)) && ((m_playerBids[teamIndex.second] == SpadesRules::NIL) || (m_playerBids[teamIndex.second] == SpadesRules::DOUBLE_NIL)))
        return true;
    else
        return false;
}

int Spades::calculateContract(pair<int, int> teamIndex) const
{
    if (bothPlayersBidNil(teamIndex))
        return 0;
    else if ((m_playerBids[teamIndex.first] == SpadesRules::NIL) || (m_playerBids[teamIndex.first] == SpadesRules::DOUBLE_NIL))
        return m_playerBids[teamIndex.second];
    else if ((m_playerBids[teamIndex.second] == SpadesRules::NIL) || (m_playerBids[teamIndex.second] == SpadesRules::DOUBLE_NIL))
        return m_playerBids[teamIndex.first];
    else
        return m_playerBids[teamIndex.first] + m_playerBids[teamIndex.second];
}

int Spades::calculateTricksWon(pair<int, int> teamIndex) const
{
    if (bothPlayersBidNil(teamIndex))
        return SpadesRules::DOUBLE_NIL;
    else if ((m_playerBids[teamIndex.first] == SpadesRules::NIL) || (m_playerBids[teamIndex.first] == SpadesRules::DOUBLE_NIL))
        return m_tricksWon[teamIndex.second];
    else if ((m_playerBids[teamIndex.second] == SpadesRules::NIL) || (m_playerBids[teamIndex.second] == SpadesRules::DOUBLE_NIL))
        return m_tricksWon[teamIndex.first];
    return m_tricksWon[teamIndex.first] + m_tricksWon[teamIndex.second];
}

void Spades::handleRoundSummary()
{
    QString message = "";

    for (int index=0; index < m_players.size(); ++ index)
    {
        // Add the contract information
        if (m_playerBids[index] == SpadesRules::NIL)
            message += QString("%1 contracted for nil ").arg(m_players[index].name());
        else if (m_playerBids[index] == SpadesRules::DOUBLE_NIL)
            message += QString("%1 contracted for double nil ").arg(m_players[index].name());
        else
            message += QString("%1 contracted for %2 tricks ").arg(m_players[index].name()).arg(m_playerBids[index]);
        // Add tricks won
        if (m_tricksWon[index] == 1)
            message += QString("and won 1 trick.\n");
        else
            message += QString("and won %1 tricks.\n").arg(m_tricksWon[index]);
    }
    // Add if the teams made their contracts
    if (m_contractMade.first)
        message += QString("\n%1-%2 team made their contract.\n").arg(m_players[Spades::PLAYER_ONE_INDEX].name()).arg(m_players[Spades::PLAYER_THREE_INDEX].name());
    else
        message += QString("\n%1-%2 team were set in their contract.\n").arg(m_players[Spades::PLAYER_ONE_INDEX].name()).arg(m_players[Spades::PLAYER_THREE_INDEX].name());
    if (m_contractMade.second)
        message += QString("%1-%2 team made their contract.\n").arg(m_players[Spades::PLAYER_TWO_INDEX].name()).arg(m_players[Spades::PLAYER_FOUR_INDEX].name());
    else
        message += QString("%1-%2 team were set in their contract.\n").arg(m_players[Spades::PLAYER_TWO_INDEX].name()).arg(m_players[Spades::PLAYER_FOUR_INDEX].name());
    // Add the scores
    message += QString("\n Resulting Scores:\n");
    message += QString(" %1-%2 Team: %3 points\n %4-%5 Team: %6 points").arg(m_players[Spades::PLAYER_ONE_INDEX].name()).arg(m_players[Spades::PLAYER_THREE_INDEX].name()).arg(m_players[Spades::PLAYER_ONE_INDEX].score()).arg(m_players[Spades::PLAYER_TWO_INDEX].name()).arg(m_players[Spades::PLAYER_FOUR_INDEX].name()).arg(m_players[Spades::PLAYER_TWO_INDEX].score());
    m_pInterface->promptMessage("Round Score", message);
}

void Spades::handleGameOver()
{
    m_pInterface->gameWon();
}

void Spades::reset()
{
    m_tricksWon.clear();
    m_rules.setSpadesBroken(false);
    m_playSequence.clear();
    // Clear any game information saved
    for (int index =0; index < m_players.size(); ++index)
        m_players[index].gameInformation().clear();
}

Spades::TeamIndexes Spades::teamWhoWon() const
{
    int teamOneScore = m_players[Spades::PLAYER_ONE_INDEX].score();
    int teamTwoScore = m_players[Spades::PLAYER_TWO_INDEX].score();

    // Make sure both teams are over the winning score
    if (! ((teamOneScore >= m_rules.winningGameScore()) && (teamTwoScore >= m_rules.winningGameScore())))
    {
        if (teamOneScore >= m_rules.winningGameScore())
            return Spades::TEAM_ONE;
        else if (teamTwoScore >= m_rules.winningGameScore())
            return Spades::TEAM_TWO;
        else
            return Spades::NON_TEAM;
    }
    else
        if (teamOneScore > teamTwoScore)
            return Spades::TEAM_ONE;
        else
            return Spades::TEAM_TWO;
    return Spades::NON_TEAM;
}
