/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 ***************************************************************************/
#ifndef SPADESRULES_H
#define SPADESRULES_H

#include "rulebase.h"

class CardSequence;

/**
 * The rules for a game of spades.
 *
 * @author John Schneiderman
 */
class SpadesRules: public RuleBase
{
public:
    /**
     * These are the special types of bids for a nil selection.
     * @param NIL is a nil bid.
     * @param DOUBLE_NIL is a bid of double nil.
     * @param NON_BID is a bid of neither nil or double nil.
     */
    enum SpecialBidTypes { NIL = 0, DOUBLE_NIL = -1, NON_BID = -3 };
    /**
     * These are some of the publicly used constants.
     * @param CARDS_TO_EXCHANGE are the cards needed to exchange on a nil or double nil bid.
     * @param MINIMUM_BID is the lowest possible bid.
     * @param MAXIMUM_BID is the highest possible bid.
     */
    enum PublicConstants { CARDS_TO_EXCHANGE = 3, MINIMUM_BID = 0, MAXIMUM_BID = 13 };
    /**
     * These are the points you get for scoring.
     * @param TRICKS_MADE_CONTRACT is the number of points made in contract for each trick.
     * @param TRICKS_EXCESS_CONTRACT is the number of points made in excess of the contract for each trick.
     * @param NIL_SCORE is the number of points made for making nil contract bid.
     * @param DOUBLE_NIL_SCORE is the number of points made for making a double nil contract bid.
     * @param PARTNERSHIP_NIL_SCORE is the number of points made for making a double nil contract bid in a partnership.
     */
    enum Scoring { TRICKS_MADE_CONTRACT = 10, TRICKS_EXCESS_CONTRACT = 1, NIL_SCORE = 100, DOUBLE_NIL_SCORE = 200, PARTNERSHIP_NIL_SCORE = 200 };

    /**
     * Default constructor.
     */
    SpadesRules();
    /**
     * Default destructor.
     */
    ~SpadesRules();
    /**
     * This determines if the conditions have been met for the game to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isGameOver(const PlayerList& players) const;
    /**
     * This determines if a card being played is a legal play.
     * @param sequence is the current play sequence.
     * @param cardPlayed is the card the player wants to player.
     * @param player is the player who played the card.
     * @return true if the card played is a legal play, false elsewise.
     */
    virtual bool isLegalPlay(const CardSequence &sequence, const Card &cardPlayed, const Player &player) const;
    /**
     * This determines if the conditions have been met for the phase to end.
     * @param players are the players in the game.
     * @param playSequence is the current play sequence.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isPhaseOver(const PlayerList &players, const CardSequence &playSequence) const;
    /**
     * This determines if the conditions have been met for the round to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isRoundOver(const PlayerList &players) const;
    /**
     * @throw KardsGTError as this method is not used.
     */
    virtual int cardValue(const Card &) const;
    /**
     * Gives the rank order value for a card.
     * @param card is the card we wish to examine.
     * @return the rank value of the card.
     */
    virtual int rankValue(const Card &card) const;
    /**
     * Gives the maximum number of players allowed in the game.
     * @return the maximum number of players.
     */
    virtual int maximumNumberOfPlayers() const;
    /**
     * Gives the minimum number of players need for the game.
     * @return the minimum number of players.
     */
    virtual int minimumNumberOfPlayers() const;
    /**
     * Gives the number of cards to deal to each player.
     * @return the number of cards to deal.
     */
    virtual int numberOfCardsToDeal(int numberOfPlayers=-1) const;
    /**
     * Gives the score needed to win the game, in this case, loose the game.
     * @return Gives the score needed to "win" the game.
     */
    virtual int winningGameScore() const;
    /**
     * This determines if spades has been broken.
     * @return true if spades has been broken in play, false elsewise.
     */
    bool isSpadesBroken() const;
    /**
     * This sets if spades has been broken.
     * @param spadesBroken true if spades has been broken, false elsewise.
     */
    void setSpadesBroken(bool spadesBroken);

private:
    /**
     * These are the number of player constants.
     * @param MINIMUM_NUMBER_OF_PLAYERS is the minimum players needed to play the game.
     * @param MAXIMUM_NUMBER_OF_PLAYERS is the maximum players needed to play the game.
     */
    enum PlayerConstants { MINIMUM_NUMBER_OF_PLAYERS = 4, MAXIMUM_NUMBER_OF_PLAYERS = 4 };
    /**
     * These are the private constants only used in this class.
     * @param CARDS_TO_DEAL is the number of cards to deal to each player.
     * @param PHASE_OVER is the number of cards in the play sequence to indicate that a phase is over.
     * @param GAME_OVER_SCORE is the score needed to pass in order to win the game.
     */
    enum PrivateConstants { CARDS_TO_DEAL = 13, PHASE_OVER = 4, GAME_OVER_SCORE = 500 };
    /// These are the card values for each card. In spades they are worth their face value. Starting with two as the lowest.
    enum CardValues { TWO=2, THREE=3, FOUR=4, FIVE=5, SIX=6, SEVEN=7, EIGHT=8, NINE=9, TEN=10, JACK=11, QUEEN=12, KING=13, ACE=14, RANK_ERR=-1 };

    /// @param m_spadesBroken is true if spades has been broken in play, false elsewise.
    bool m_spadesBroken;
};
#endif
