/***************************************************************************
 *   Copyright (C) 2006 by John Schneiderman                               *
 *   JohnMS@member.fsf.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "normanheartsai.h"
#include "cardproperties.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"

#include <vector>
using std::vector;

NormanHeartsAI::NormanHeartsAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand)
{
    m_playSequence = playSequence;
    m_pRules = &rules;
    m_hand = hand;
}

NormanHeartsAI::~NormanHeartsAI()
{}

CardSequence NormanHeartsAI::selectCards() const
{
    CardSequence cards;
    CardProperties properties(m_hand);

    if (m_hand.hasCard(Card(Card::TWO, Card::CLUBS))) // Are we to start the round
        cards.addCard(Card(Card::TWO, Card::CLUBS));
    else if (m_playSequence.isEmpty()) // Are we leading the phase?
    {
        // Choose a suit with a low card to lead with
        CardSequence hearts, clubs, spades, diamonds, lowCards;

        // Find out the number of cards in each suit
        hearts = properties.suits(Card::HEARTS);
        clubs = properties.suits(Card::CLUBS);
        spades = properties.suits(Card::SPADES);
        diamonds = properties.suits(Card::DIAMONDS);

        // Find lowest card for each suit
        lowCards.addCard(lowestCard(hearts));
        lowCards.addCard(lowestCard(clubs));
        lowCards.addCard(lowestCard(spades));
        lowCards.addCard(lowestCard(diamonds));

        // Play the lowest card we found
        cards.addCard(lowestCard(lowCards));
    }
    else
    {
        Card::Suit playingSuit=m_playSequence.front().suit();
        CardSequence cardSuits = properties.suits(playingSuit);

        if (! cardSuits.isEmpty()) // Do we have a suit to play
        {
            if (m_playSequence.hasCard(Card(Card::QUEEN, Card::SPADES)))
                cards.addCard(lowestCard(cardSuits)); // Play our lowest card to try and avoid the QS.
            else
                cards.addCard(highestCard(cardSuits)); // Play our high cards to save our low cards for the later in the round.
        }
        else
        {
            // Do we have the QS or hearts?
            if (m_hand.hasCard(Card(Card::QUEEN, Card::SPADES)))
                cards.addCard(Card(Card::QUEEN, Card::SPADES));
            else
            {
                cardSuits = properties.suits(Card::HEARTS);

                if (! cardSuits.isEmpty())
                    cards.addCard(highestCard(cardSuits));
                else
                    cards.addCard(findHighestCardToPlay()); // Play the highest card from some suit
            }
        }
    }
    return cards;
}

CardSequence NormanHeartsAI::passCards() const
{
    CardSequence cards;
    CardProperties properties(m_hand);
    CardSequence hearts, clubs, spades, diamonds;

    // Find out the number of suits
    hearts = properties.suits(Card::HEARTS);
    clubs = properties.suits(Card::CLUBS);
    spades = properties.suits(Card::SPADES);
    diamonds = properties.suits(Card::DIAMONDS);

    // Discard all our high spades (A K Q) if we have less than 4 spades in our hand
    if (spades.size() < 4)
    {
        for (int index=0; index < spades.size(); ++index)
            if (spades[index].rank() == Card::ACE)
                cards.addCard(spades[index]);
            else if (spades[index].rank() == Card::KING)
                cards.addCard(spades[index]);
            else if (spades[index].rank() == Card::QUEEN)
                cards.addCard(spades[index]);
    }

    // Discard hearts if they are unguarded
    if (cards.size() < 3)
        if (! isGuarded(hearts)) // Remove unguarded hearts
            addUnguardedHighCards(cards, hearts);

    // Discard the remaining unguarded high cards.
    if (cards.size() < 3)
    {
        if (! isGuarded(clubs)) // Remove unguarded clubs
            addUnguardedHighCards(cards, clubs);
        if (cards.size() < 3) // Remove unguarded diamonds
            if (! isGuarded(diamonds))
                addUnguardedHighCards(cards, diamonds);
    }

    // Do we have a long (4 or more) suit
    if (cards.size() < 3)
    {
        if (spades.size() > 4) // Lower the spades
            shaveLargeSuits(cards, spades);
        if (cards.size() < 3) // Lower the hearts
            if (hearts.size() > 4)
                shaveLargeSuits(cards, hearts);
        if (cards.size() < 3) // Lower the clubs
            if (clubs.size() > 4)
                shaveLargeSuits(cards, clubs);
        if (cards.size() < 3) // Lower the diamonds
            if (diamonds.size() > 4)
                shaveLargeSuits(cards, diamonds);
    }

    // Do we have no other strategies left?
    if (cards.size() < 3)
    {
        BasicGameStrategies gameai(*m_pRules);
        Card testCard;

        testCard=gameai.randomCardsWithNoLegalChecks(1, m_hand).front();
        while (cards.size() < 3)
        {
            if (! cards.hasCard(testCard))
                cards.addCard(testCard);
            testCard=gameai.randomCardsWithNoLegalChecks(1, m_hand).front();
        }
    }
    return cards;
}

bool NormanHeartsAI::isGuarded(const CardSequence &sequence) const
{
    bool guarded=false;
    int lowCards=0;

    for (int index=0; index < sequence.size(); ++index)
        if ((sequence[index].rank() < Card::EIGHT) && (sequence[index].rank() != Card::ACE)) // Less than 8 and not Ace
        {
            ++lowCards;
            if (lowCards > 2)
                guarded = true;
        }
    return guarded;
}

void NormanHeartsAI::addUnguardedHighCards(CardSequence &sequence, const CardSequence &toGuard) const
{
    for (int index=toGuard.size() - 1; index >=0; --index) // Remove the remaining high cards
    {
        if ((toGuard[index].rank() >= Card::EIGHT) || (toGuard[index].rank() == Card::ACE))
            sequence.addCard(toGuard[index]);
        if (sequence.size() == 3)
            break;
    }
}

void NormanHeartsAI::shaveLargeSuits(CardSequence &sequence, const CardSequence &shaveSuit) const
{
    for (int index=shaveSuit.size() - 1; index >= 0; --index)
        if (! sequence.hasCard(shaveSuit[index]))
        {
            sequence.addCard(shaveSuit[index]);
            if (sequence.size() == 3)
                break;
        }
}

Card NormanHeartsAI::lowestCard(const CardSequence &sequence) const
{
    if (sequence.isEmpty())
        return Card();
    const Card *low=&sequence[0];

    // Play our high cards to save our low cards for the later in the round.
    for (int index=1; index < sequence.size(); ++index)
        if (((sequence[index].rank() < low->rank()) && (sequence[index].rank() != Card::ACE)) || ((low->rank() == Card::ACE)))
            low = &sequence[index];
    return *low;
}

Card NormanHeartsAI::highestCard(const CardSequence &sequence) const
{
    if (sequence.isEmpty())
        return Card();
    const Card *high=&sequence[0];

    for (int index=1; index < sequence.size(); ++index)
        if (high->rank() == Card::RANK_ERR)
            high = &sequence[index];
        else if (((sequence[index].rank() > high->rank()) || (sequence[index].rank() == Card::ACE)) && (sequence[index].rank() != Card::RANK_ERR))
            high = &sequence[index];
    if (high->rank() == Card::RANK_ERR)
        throw KardsGTError("NormanHeartsAI", "highestCard", "Failed to find a valid high card!");
    return *high;
}

Card NormanHeartsAI::findHighestCardToPlay() const
{
    CardSequence hearts, clubs, spades, diamonds, highCards;
    CardProperties properties(m_hand);

    // Find out the number of cards in each suit
    hearts = properties.suits(Card::HEARTS);
    clubs = properties.suits(Card::CLUBS);
    spades = properties.suits(Card::SPADES);
    diamonds = properties.suits(Card::DIAMONDS);

    // Find highest card for each suit
    highCards.addCard(highestCard(hearts));
    highCards.addCard(highestCard(clubs));
    highCards.addCard(highestCard(spades));
    highCards.addCard(highestCard(diamonds));

    return highestCard(highCards);
}
