/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CARDDECK_H
#define CARDDECK_H

#include "cardpile.h"
#include "card.h"

#include <vector>
using std::vector;

/**
 * This is a deck of cards.
 *
 * @author John Schneiderman
 */
class CardDeck: public CardPile
{
public:
    /**
     * This constructor creates a full-ready deck.
     * @param numberOfDecks is how many card decks to create in the deck.
     */
    CardDeck(int numberOfDecks = 1);
    /**
     * This constructor creates a full-ready deck.
     * @param cards are the cards to exclude from the deck.
     * @param numberOfDecks is how many card decks to create in the deck.
     */
    CardDeck(vector<Card> cards, int numberOfDecks = 1);
    /**
     * Fills any missing cards in the deck, and then shuffles the deck.
     */
    void shuffle();
    /**
     * This removes a card off the top of the deck.
     * @return the card removed from the top of the deck.
     */
    Card dealCard();
    /**
     * This randomly cuts the deck, and places the bottom half on top of the other half.
     * @note That the cut is guaranteed to be at least GUARANTEE_CUT_SIZE.
     */
    void cutDeck();
    /**
     * This removes all cards of a specified rank from the deck.
     * @param rank is the ranks to remove.
     * @note When this method is called, the deck is recreated.
     */
    void removeCards(Card::Rank rank);
    /**
     * This removes all cards of a specified card from the deck.
     * @param card is the card to remove.
     * @note When this method is called, the deck is recreated.
     */
    void removeCards(const Card &card);
    /**
     * This will add a joker card to the deck.
     * @note when this method is called, the deck is recreated, and the number of jokers is cumulative.
     */
    void addJoker();
    /**
     * This gives the number of decks current used in the deck.
     * @return the number of decks used.
     */
    int numberOfDecks() const;
    /**
     * This sets how many card decks to create in the deck.
     * @param numberOfDecks is number of decks to create.
     * @note When this method is called, the deck is recreated.
     */
    void setNumberOfDecks(int numberOfDecks);

protected:
    /**
     * This creates an ordered deck of cards based the on suit and rank orders listed in the @ref Card class.
     * @note This will clear the current deck.
     * @note This will initialise m_deckSize.
     */
    void createDeck();
    /**
     * Finds all the missing cards in the deck.
     * @return the missing cards from the deck.
     */
    CardSequence findMissingCards() const;

private:
    /**
     * The constants used in the card deck.
     * @param GUARANTEE_CUT_SIZE is the minimum number of cards that will be used when cutting the deck.
     */
    enum PrivateConstants
    {
        GUARANTEE_CUT_SIZE = 13
    };
    /// @param m_removedCards is the cards removed from the deck.
    vector<Card> m_removedCards;
    /// @param m_numberOfDecks is the number of decks in this deck.
    int m_numberOfDecks;
    /// @param m_numberOfJokers is the number of jokers in this deck.
    int m_numberOfJokers;
    /// @param m_deckSize is the expected maximum number of cards found in a deck.
    int m_deckSize;


    /**
     * Determines if a card is allowed in the deck
     * @return true if the card is allowed, false if it is not.
     */
    bool allowedCard(const Card &card) const;
    /**
     * Inserts all jokers into the bottom of the current deck.
     * @param pile is the pile of cards to insert the jokers into.
     */
    void insertJokers(CardSequence &pile) const;
};
#endif
