/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "player.h"
#include "card.h"
#include "kardsgterror.h"

#include <QTextStream>

Player::Player(): m_hand(), m_gameInformation()
{
    m_name="Not Known";
    m_gamesWon=m_score=0;
    m_level=Skill_Error;
    m_dealer=m_turn=false;
}

Player::~Player()
{}

QString Player::name() const
{
    return m_name;
}

void Player::setName(const QString &name)
{
    m_name=name;
}

const CardSequence& Player::hand() const
{
    return m_hand;
}

CardSequence& Player::hand()
{
    return m_hand;
}

int Player::score() const
{
    return m_score;
}

void Player::addPoint(int point)
{
    m_score += point;
}

void Player::removePoint(int point)
{
    m_score -= point;
}

bool Player::isDealer() const
{
    return m_dealer;
}

void Player::setDealer(bool m_dealer)
{
    this->m_dealer=m_dealer;
}

void Player::incrementGamesWon()
{
    m_gamesWon++;
}

void Player::decrementGamesWon()
{
    m_gamesWon--;
}

int Player::gamesWon() const
{
    return m_gamesWon;
}

bool Player::isTurn() const
{
    return m_turn;
}

void Player::setTurn(bool turn)
{
    m_turn=turn;
}

Player::SkillLevel Player::skillLevel() const
{
    return m_level;
}

void Player::setSkillLevel(Player::SkillLevel level)
{
    m_level=level;
}

bool Player::operator==(const Player &player) const
{
    if (m_name == player.name())
        if (m_hand == player.hand())
            return true;
        else
            return false;
    else
        return false;
}

void Player::reset()
{
    m_dealer=false;
    m_turn=false;
    m_hand.clear();
}

void Player::resetRoundScore()
{
    m_score=0;
}

void Player::clear()
{
    reset();
    resetRoundScore();
    m_gamesWon=0;
    m_gameInformation.clear();
}

GameInformation Player::gameInformation() const
{
    return m_gameInformation;
}

GameInformation& Player::gameInformation()
{
    return m_gameInformation;
}

QTextStream& operator<<(QTextStream &out, const Player &player)
{
    QString name;

    name = player.m_name;
    name.replace(" ", "_");
    out << "Name: " << name << endl;
    out << "Hand: " << player.m_hand << endl;
    out << "Score: " << player.m_score << endl;
    out << "GamesWon: " << player.m_gamesWon << endl;
    out << "Dealer: " << player.m_dealer << endl;
    out << "Turn: " << player.m_turn << endl;
    out << "Level: " << player.m_level << endl;
    out << "GameInformation: " << player.m_gameInformation;
    return out;
}

ostream& operator<<(ostream &out, const Player &player)
{
    QString temp;

    temp = player.m_name;
    temp.replace(" ", "_");
    out << "Name: " << temp.toStdString() << std::endl;
    out << "Hand: " << player.m_hand << std::endl;
    out << "Score: " << player.m_score << std::endl;
    out << "GamesWon: " << player.m_gamesWon << std::endl;
    out << "Dealer: " << player.m_dealer << std::endl;
    out << "Turn: " << player.m_turn << std::endl;
    out << "Level: " << player.m_level << std::endl;
    out << "GameInformation: " << player.m_gameInformation;
    return out;
}

QTextStream& operator>>(QTextStream &in, Player &player)
{
    int intToBool;
    QString label;

    in >> label >> player.m_name;
    player.m_name = player.m_name.replace("_", " ");
    in >> label >> player.m_hand;
    in >> label >> player.m_score;
    in >> label >> player.m_gamesWon;
    in >> label >> intToBool;
    player.m_dealer=static_cast<bool>(intToBool);
    in >> label >> intToBool;
    player.m_turn=static_cast<bool>(intToBool);
    // Read in the saved level, but only use it if the value isn't known yet.
    in >> label >> intToBool;
    if (player.m_level == Player::Skill_Error)
        player.m_level=static_cast<Player::SkillLevel>(intToBool);
    in >> label >> player.m_gameInformation;
    return in;
}
