/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CRAZY_EIGHTS_RULES_H
#define CRAZY_EIGHTS_RULES_H

#include "rulebase.h"

class CardSequence;
class CardPile;

/**
 * The rules for a game of Crazy Eights.
 *
 * @author John Schneiderman
 */
class CrazyEightsRules : public RuleBase
{
public:
    /**
     * These are the public rule constants.
     */
    enum PublicConstants
    {
        NUMBER_OF_PLAYERS_FOR_TWO_DECKS = 5,
        NUMBER_OF_DECKS_FOR_LARGE_GROUP = 2
    };

    /**
     * Default constructor.
     */
    CrazyEightsRules();
    /**
     * Default destructor.
     */
    ~CrazyEightsRules();
    /**
     * This determines if the conditions have been met for the game to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isGameOver(const PlayerList& players) const;
    /**
     * This determines if a card being played is a legal play.
     * @param playSequence is the current play sequence.
     * @param cardPlayed is the card the player wants to player.
     * @param player is the player who played the card.
     * @return true if the card played is a legal play, false elsewise.
     */
    virtual bool isLegalPlay(const CardSequence& playSequence, const Card& cardPlayed, const Player& player) const;
    /**
     * This determines if the conditions have been met for the phase to end.
     * @param players are the players in the game.
     * @param playSequence is the current play sequence.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isPhaseOver(const PlayerList &players, const CardSequence &playSequence) const;
    /**
     * This determines if the conditions have been met for the round to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isRoundOver(const PlayerList& players) const;
    /**
     * This gives the card point value.
     * @param card is the card we want to know the value of.
     * @return the point value of the card.
     */
    virtual int cardValue(const Card& card) const;
    /**
     * Gives the rank order value for a card.
     * @param card is the card we wish to examine.
     * @return the rank value of the card.
     */
    virtual int rankValue(const Card& card) const;
    /**
     * Gives the maximum number of players allowed in the game.
     * @return the maximum number of players.
     */
    virtual int maximumNumberOfPlayers() const;
    /**
     * Gives the minimum number of players need for the game.
     * @return the minimum number of players.
     */
    virtual int minimumNumberOfPlayers() const;
    /**
     * Gives the number of cards to deal to each player.
     * @param numberOfPlayers is the number of players in the game.
     * @return the number of cards to deal.
     */
    virtual int numberOfCardsToDeal(int numberOfPlayers = -1) const;
    /**
     * Gives the score needed to win the game.
     * @param numberOfPlayers is the number of players in the game.
     * @return Gives the score needed to "win" the game.
     */
    virtual int winningGameScore(int numberOfPlayers = -1) const;
    /**
     * This determines if no one can play a card, called a block.
     * @param players are all the players in the game.
     * @param deck is the current deck of cards in the game.
     * @param playSequence is the current discard sequence.
     * @return true if no one can play a card, and false elsewise.
     */
    bool isRoundOverFromBlock(const PlayerList &players, const CardPile &deck, const CardSequence& playSequence) const;

private:
    /**
     * These are the private constants used in the class.
     *
     * @param MIN_PLAYERS is the minimum players needed to play the game.
     * @param MAX_PLAYERS is the maximum players needed to play the game.
     * @param CARDS_FOR_TWO is the number of cards to deal for two players.
     * @param CARDS_FOR_THREE_TO_SEVEN is the number of cards to deal for more than two players.
     * @param WINNING_POINTS_PER_PLAYER is the number of points needed to win per player.
     */
    enum PrivateConstants
    {
        MIN_PLAYERS=2,
        MAX_PLAYERS=7,
        CARDS_FOR_TWO=7,
        CARDS_FOR_THREE_TO_SEVEN=5,
        WINNING_POINTS_PER_PLAYER=50
    };
    /**
     * The points for a card at the end of a round.
     *
     * @param EIGHT is the points scored for an eight card in the hand.
     * @param ACE is the points scored for an ace card in the hand.
     * @param FACE_CARD is the points cored for a face card in the hand.
     */
    enum ScorePoints
    {
        EIGHT=50,
        ACE=1,
        FACE_CARD=10
    };

    /**
     * This determines if a player has a legal play.
     * @param hand if the players current hand.
     * @param playSequence is the current discard sequence.
     * @param player is the player to examine.
     * @return true if the player has a card he can play legally, false elsewise.
     */
    bool playerHasLegalPlay(const CardSequence &hand, const CardSequence& playSequence, const Player &player) const;
};
#endif
