/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef HEARTSRULES_H
#define HEARTSRULES_H

#include "rulebase.h"

class CardSequence;

/**
 * The rules for a game of hearts.
 *
 * @author John Schneiderman
 */
class HeartsRules: public RuleBase
{
public:
    /**
     * These are the constants used through-out the hearts game.
     * @param PASS_TO_PLAYER are the number of cards to pass to a player at the beginning of a round.
     * @param SHOT_THE_MOON is the number of points earned when a player collects all hearts and the QS.
     */
    enum PublicConstants
    {
        PASS_TO_PLAYER = 3,
        SHOT_THE_MOON = 26
    };

    /**
     * Default constructor.
     */
    HeartsRules();
    /**
     * Default destructor.
     */
    ~HeartsRules();
    /**
     * This determines if the conditions have been met for the game to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isGameOver(const PlayerList& players) const;
    /**
     * This determines if a card being played is a legal play.
     * @param sequence is the current play sequence.
     * @param cardPlayed is the card the player wants to player.
     * @param player is the player who played the card.
     * @return true if the card played is a legal play, false elsewise.
     */
    virtual bool isLegalPlay(const CardSequence& sequence, const Card& cardPlayed, const Player& player) const;
    /**
     * This determines if the conditions have been met for the round to end.
     * @param players are the players in the game.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isRoundOver(const PlayerList& players) const;
    /**
     * This determines if the conditions have been met for the phase to end.
     * @param players are the players in the game.
     * @param playSequence is the current play sequence.
     * @return true if the conditions have been met, false elsewise.
     */
    virtual bool isPhaseOver(const PlayerList &players, const CardSequence &playSequence) const;
    /**
     * This gives the card point value.
     * @param card is the card we want to know the value of.
     * @return the point value of the card.
     */
    virtual int cardValue(const Card& card) const;
    /**
     * Gives the maximum number of players allowed in the game.
     * @return the maximum number of players.
     */
    virtual int maximumNumberOfPlayers() const;
    /**
     * Gives the minimum number of players need for the game.
     * @return the minimum number of players.
     */
    virtual int minimumNumberOfPlayers() const;
    /**
     * Gives the number of cards to deal to each player.
     * @return the number of cards to deal.
     */
    virtual int numberOfCardsToDeal(int numberOfPlayers=-1) const;
    /**
     * Gives the score needed to win the game, in this case, loose the game.
     * @return Gives the score needed to "win" the game.
     */
    virtual int winningGameScore(int numberOfPlayers = -1) const;
    /**
     * Searches through a sequence and finds the cards that are worth points.
     * @param playSequence is the current play sequence.
     * @return the cards found to be worth something.
     */
    CardSequence cardPointers(const CardSequence &playSequence) const;
    /**
     * Gives the rank value for a card.
     * @param card is the card we wish to examine.
     * @return the rank value of the card.
     */
    virtual int rankValue(const Card &card) const;
    /**
     * Determines if a card sequence shoots the moon.
     * @param sequence is the sequence of cards to examine.
     * @note Sequence must only contain point counting hearts
     * @return true if the sequence shoots the moon, false elsewise.
     */
    bool hasShotTheMoon(const CardSequence &sequence) const;

private:
    /**
     * These are the constants used through-out the rules of hearts.
     * @param MAXIMUM_NUMBER_OF_PLAYERS is the maximum number of players allowed in a game of hearts.
     * @param MINIMUM_NUMBER_OF_PLAYERS is the minimum number of players needed in a game of hearts.
     * @param WINNING_GAME_SCORE is the score needed to win the game, in this case to actually lose the game.
     * @param NUMBER_OF_CARDS_TO_DEAL is the number of cards to deal to each player.
     * @param PHASE_OVER is the number of cards needed for a phase to be done.
     * @param NUMBER_OF_CARDS_TO_SHOOT_THE_MOON is the number of cards needed to shoot the moon. (13 hearts + QS)
     */
    enum PrivateConstants
    {
        MAXIMUM_NUMBER_OF_PLAYERS = 4,
        MINIMUM_NUMBER_OF_PLAYERS = 4,
        WINNING_GAME_SCORE = 100,
        NUMBER_OF_CARDS_TO_DEAL = 13,
        PHASE_OVER = 4,
        NUMBER_OF_CARDS_TO_SHOOT_THE_MOON=14
    };
    /// These are the card values for each card. In hearts they are worth their face value. Starting with two as the lowest.
    enum CardValues
    {
        TWO=2,
        THREE=3,
        FOUR=4,
        FIVE=5,
        SIX=6,
        SEVEN=7,
        EIGHT=8,
        NINE=9,
        TEN=10,
        JACK=11,
        QUEEN=12,
        KING=13,
        ACE=14,
        RANK_ERR=-1
    };

};
#endif
